<?php

namespace App\Http\Controllers\Traits;

use App\Models\Project;
use App\Models\Defect;
use App\Models\RequiredDocument;
use App\Models\SubcontractorEmployeeDocument;
use App\Models\InductionDocument;
use App\Models\InductionDocumentSignature;
use Carbon\Carbon;

trait SubcontractorPortalTrait
{
    protected function getEmployeeManagementProjectsData($responseData, $employee, $employeeMeta, $subcontractorCompanies)
    {
        $employeeProjectIds = $employeeMeta->project_ids ?? [];
        
        if (empty($employeeProjectIds)) {
            $responseData['assigned_projects'] = [];
            return $responseData;
        }

        // Get all projects assigned to this employee
        $projects = Project::whereIn('id', $employeeProjectIds)
            ->where('is_deleted', 0)
            ->with(['company:id,name'])
            ->get();

        // Build project company map
        $projectCompanyMap = [];
        foreach ($subcontractorCompanies as $company) {
            $projectIds = $company->project_ids ?? [];
            foreach ($projectIds as $projectId) {
                if (in_array($projectId, $employeeProjectIds) && !isset($projectCompanyMap[$projectId])) {
                    $projectCompanyMap[$projectId] = [
                        'customer_id' => $company->customer_id,
                        'workspace_id' => $company->workspace_id,
                    ];
                }
            }
        }

        // Count defects per project
        $assignedProjects = [];
        foreach ($projects as $project) {
            $customerId = $projectCompanyMap[$project->id]['customer_id'] ?? null;
            $workspaceId = $projectCompanyMap[$project->id]['workspace_id'] ?? null;

            $defectsCount = 0;
            if ($customerId && $workspaceId) {
                $defectsCount = Defect::where('project_id', $project->id)
                    ->where('assigned_subcontractor_emp_id', $employee->id)
                    ->where('assignment_type', 'subcontractor_employee')
                    ->where('customer_id', $customerId)
                    ->where('workspace_id', $workspaceId)
                    ->where('del', '0')
                    ->count();
            }

            $assignedProjects[] = [
                'id' => $project->id,
                'project_name' => $project->title,
                'company_name' => $project->company ? $project->company->name : null,
                'defects_assigned' => $defectsCount,
            ];
        }

        $responseData['assigned_projects'] = $assignedProjects;
        return $responseData;
    }

    protected function getEmployeeManagementDefectsData($responseData, $employee, $subcontractorCompanies)
    {
        // Collect all defects assigned to this employee across all companies
        $allDefects = collect();
        
        foreach ($subcontractorCompanies as $company) {
            $defects = Defect::where('assigned_subcontractor_emp_id', $employee->id)
                ->where('assignment_type', 'subcontractor_employee')
                ->where('customer_id', $company->customer_id)
                ->where('workspace_id', $company->workspace_id)
                ->where('del', '0')
                ->with(['project:id,title', 'site:id,title'])
                ->orderBy('created_at', 'desc')
                ->get();
            
            $allDefects = $allDefects->merge($defects);
        }

        $defectsData = $allDefects->map(function ($defect) {
            $dueDate = null;
            if ($defect->due_date) {
                $dueDate = is_string($defect->due_date) 
                    ? Carbon::parse($defect->due_date)->format('Y-m-d') 
                    : $defect->due_date->format('Y-m-d');
            }

            return [
                'id' => $defect->id,
                'title' => $defect->title,
                'description' => $defect->description,
                'priority' => $defect->priority,
                'status' => $defect->status,
                'due_date' => $dueDate,
                'project' => $defect->project ? [
                    'id' => $defect->project->id,
                    'title' => $defect->project->title,
                ] : null,
                'site' => $defect->site ? [
                    'id' => $defect->site->id,
                    'title' => $defect->site->title,
                ] : null,
                'created_at' => $defect->created_at,
            ];
        })->values();

        $responseData['defects'] = $defectsData;
        return $responseData;
    }

    protected function getEmployeeManagementDocumentsData($responseData, $employee, $subcontractorCompanies)
    {
        // Collect all required documents and uploaded documents across all companies
        $allRequiredDocuments = collect();
        $allUploadedDocuments = collect();
        foreach ($subcontractorCompanies as $company) {
            // Get required documents for external/subcontractor employees
            $requiredDocs = RequiredDocument::where('customer_id', $company->customer_id)
                ->where('workspace_id', $company->workspace_id)
                ->where('del', '0')
                ->where(function ($q) {
                    $q->where('for_external', true);
                })
                ->with(['requiredDocumentActiveField'])
                ->get();

            $allRequiredDocuments = $allRequiredDocuments->merge($requiredDocs);

            // Get uploaded documents for this employee
            $uploadedDocs = SubcontractorEmployeeDocument::where('employee_id', $employee->id)
                ->where('customer_id', $company->customer_id)
                ->where('workspace_id', $company->workspace_id)
                ->where('del', '0')
                ->with(['requiredDocument', 'requiredDocumentField', 'approver'])
                ->get();

            $allUploadedDocuments = $allUploadedDocuments->merge($uploadedDocs);
        }

        // Group uploaded documents by required_document_id
        $uploadedByDocId = $allUploadedDocuments->groupBy('required_document_id');

        // Build documents with status
        $documentsData = $allRequiredDocuments->map(function ($doc) use ($uploadedByDocId, $employee) {
            $uploaded = $uploadedByDocId->get($doc->id) ?? collect();

            // Map uploaded values to fields
            $uploadedFieldValues = [];
            foreach ($uploaded as $uploadedDoc) {
                $uploadedFieldValues[$uploadedDoc->required_document_field_id] = [
                    'id' => $uploadedDoc->id,
                    'value' => $uploadedDoc->value,
                    'approval_status' => $uploadedDoc->approval_status,
                    'rejection_reason' => $uploadedDoc->rejection_reason,
                    'approved_by' => $uploadedDoc->approver ? $uploadedDoc->approver->name : null,
                    'approved_at' => $uploadedDoc->approved_at,
                    'updated_at' => $uploadedDoc->updated_at,
                ];
            }

            $isComplete = true;

            $fields = $doc->requiredDocumentActiveField ?? collect();
            
            $requiredFields = $fields->where('field_required', true);
            $optionalFields = $fields->where('field_required', false);
            
            $hasAnySubmittedValue = false;
            
            // Detect if ANY field has a submitted value
            foreach ($fields as $field) {
                if (
                    isset($uploadedFieldValues[$field->id]) &&
                    !empty($uploadedFieldValues[$field->id]['value'])
                ) {
                    $hasAnySubmittedValue = true;
                    break;
                }
            }
            
            /**
             * CASE 1: No fields
             */
            if ($fields->isEmpty()) {
                $isComplete = true;
            }
            
            /**
             * CASE 2: Only optional fields
             * → Any ONE submitted value makes it complete
             */
            elseif ($requiredFields->isEmpty()) {
                $isComplete = $hasAnySubmittedValue;
            }
            
            /**
             * CASE 3: Required + optional fields
             * → ALL required fields must be submitted
             */
            else {
                foreach ($requiredFields as $field) {
                    if (
                        !isset($uploadedFieldValues[$field->id]) ||
                        empty($uploadedFieldValues[$field->id]['value'])
                    ) {
                        $isComplete = false;
                        break;
                    }
                }
            }

            // Calculate document-level approval status
            $docApprovalStatus = null;
            $docRejectionReason = null;
            $docApprovedBy = null;
            $docApprovedAt = null;

            if ($uploaded->isNotEmpty()) {
                $allApproved = $uploaded->every(fn($d) => $d->approval_status === 'approved');
                $anyRejected = $uploaded->contains(fn($d) => $d->approval_status === 'rejected');

                if ($allApproved) {
                    $docApprovalStatus = 'approved';
                    $firstApproved = $uploaded->first();
                    $docApprovedBy = $firstApproved->approver ? $firstApproved->approver->name : null;
                    $docApprovedAt = $firstApproved->approved_at;
                } elseif ($anyRejected) {
                    $docApprovalStatus = 'rejected';
                    $rejectedDoc = $uploaded->firstWhere('approval_status', 'rejected');
                    $docRejectionReason = $rejectedDoc->rejection_reason;
                    $docApprovedBy = $rejectedDoc->approver ? $rejectedDoc->approver->name : null;
                    $docApprovedAt = $rejectedDoc->approved_at;
                } else {
                    $docApprovalStatus = 'pending';
                }
            }

            return [
                'id' => $doc->id,
                'title' => $doc->title,
                'is_complete' => $isComplete,
                'approval_status' => $docApprovalStatus,
                'rejection_reason' => $docRejectionReason,
                'approved_by' => $docApprovedBy,
                'approved_at' => $docApprovedAt,
                'fields' => $doc->requiredDocumentActiveField ? $doc->requiredDocumentActiveField->map(function ($field) use ($uploadedFieldValues) {
                    $fieldId = $field->id;
                    $uploadedValue = $uploadedFieldValues[$fieldId] ?? null;

                    return [
                        'id' => $field->id,
                        'field_name' => $field->field_name,
                        'field_type' => $field->field_type,
                        'field_required' => $field->field_required,
                        'priority' => $field->priority,
                        'uploaded_value' => $uploadedValue ? $uploadedValue['value'] : null,
                        'uploaded_id' => $uploadedValue ? $uploadedValue['id'] : null,
                        'approval_status' => $uploadedValue ? $uploadedValue['approval_status'] : null,
                        'rejection_reason' => $uploadedValue ? $uploadedValue['rejection_reason'] : null,
                        'approved_by' => $uploadedValue ? $uploadedValue['approved_by'] : null,
                        'approved_at' => $uploadedValue ? $uploadedValue['approved_at'] : null,
                        'last_updated' => $uploadedValue ? $uploadedValue['updated_at'] : null,
                    ];
                }) : collect(),
            ];
        })->values();

        $responseData['documents'] = $documentsData;
        return $responseData;
    }

    protected function getEmployeeManagementInductionData($responseData, $employee, $subcontractorCompanies)
    {
        // Collect all induction documents across all companies
        $allInductionDocuments = collect();
        $allSignatures = collect();

        foreach ($subcontractorCompanies as $company) {
            // Get all active induction documents
            $inductionDocuments = InductionDocument::where('customer_id', $company->customer_id)
                ->where('workspace_id', $company->workspace_id)
                ->where('del', '0')
                ->where('is_active', true)
                ->orderBy('title')
                ->orderBy('version', 'desc')
                ->get();

            // Filter documents that include 'external' or 'subcontractor' in role_types or 'all'
            $filteredDocs = $inductionDocuments->filter(function ($doc) {
                $roleTypes = $doc->role_types ?? [];
                return in_array('all', $roleTypes) || in_array('external', $roleTypes) || in_array('subcontractor', $roleTypes);
            });

            $allInductionDocuments = $allInductionDocuments->merge($filteredDocs);

            // Get all signatures for this employee
            $signatures = InductionDocumentSignature::where('employee_id', $employee->id)
                ->where('signature_type', 'subcontractor_employee')
                ->where('customer_id', $company->customer_id)
                ->where('workspace_id', $company->workspace_id)
                ->where('del', '0')
                ->get();

            $allSignatures = $allSignatures->merge($signatures);
        }

        // Group by title and get latest version for each
        $documentsByTitle = [];
        foreach ($allInductionDocuments as $doc) {
            $title = $doc->title;
            if (!isset($documentsByTitle[$title]) || $doc->version > $documentsByTitle[$title]->version) {
                $documentsByTitle[$title] = $doc;
            }
        }

        // Build induction documents with signature status
        $inductionData = collect($documentsByTitle)->map(function ($doc) use ($allSignatures, $employee) {
            // Find valid signature for this document version
            $signature = $allSignatures->first(function ($sig) use ($doc) {
                return $sig->induction_document_id == $doc->id 
                    && $sig->document_version == $doc->version 
                    && $sig->is_valid == true;
            });

            // If not found and it's a minor update, check for signature on any previous version
            if (!$signature && $doc->update_type === 'minor') {
                $rootDocumentId = $doc->id;
                $currentDoc = $doc;
                while ($currentDoc && $currentDoc->parent_document_id) {
                    $rootDocumentId = $currentDoc->parent_document_id;
                    $currentDoc = \App\Models\InductionDocument::find($rootDocumentId);
                }
                
                $chainIds = \App\Models\InductionDocument::where(function($q) use ($rootDocumentId) {
                    $q->where('id', $rootDocumentId)
                      ->orWhere('parent_document_id', $rootDocumentId);
                })->pluck('id')->toArray();
                
                // Now check if we have a signature for any of these
                $signature = $allSignatures->first(function ($sig) use ($chainIds) {
                    return in_array($sig->induction_document_id, $chainIds) && $sig->is_valid;
                });
            }

            $hasSigned = $signature !== null;

            return [
                'id' => $doc->id,
                'title' => $doc->title,
                'version' => $doc->version,
                'description' => $doc->description,
                'file' => $doc->file ? url($doc->file) : null,
                'has_signed' => $hasSigned,
                'signed_at' => $hasSigned ? $signature->signed_at : null,
                'signature_path' => $hasSigned && $signature->signature_path ? url($signature->signature_path) : null,
                'signed_file' => $hasSigned && $signature->signed_file ? url($signature->signed_file) : null,
                'update_type' => $doc->update_type,
            ];
        })->values();

        $responseData['induction_documents'] = $inductionData;
        return $responseData;
    }
}
