<?php

namespace App\Http\Controllers\Traits;

use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;

trait SubcontractorProfileTrait
{

    protected function updateSubcontractorProfile(Request $request)
    {
        // Get authenticated user
        $user = Auth::user();
        
        // Check if user is subcontractor (user_type = 5)
        if ($user->user_type != config('constants.user_types.subcontractor')) {
            return $this->message('Unauthorized access', 403);
        }
        
        // Validate request
        $validator = $this->subcontractorProfileValidationRequest($request);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        
        $validatedData = $validator->validated();
        
        // Handle password update if provided
        if (!empty($validatedData['old_password'])) {
            // Verify old password
            if (!Hash::check($validatedData['old_password'], $user->password)) {
                return $this->message('Current password is incorrect', 422);
            }
            
            // Update password with new password
            $validatedData['password'] = Hash::make($validatedData['new_password']);
        }
        
        // Remove password fields from validated data (we don't want to store them as plain text)
        unset($validatedData['old_password'], $validatedData['new_password'], $validatedData['new_password_confirmation']);
        
        // Check email uniqueness if email is provided and different from current email
        if (isset($validatedData['email']) && $validatedData['email'] !== $user->email) {
            if (User::where('email', $validatedData['email'])->where('id', '!=', $user->id)->exists()) {
                return $this->message('Email already exists', 422);
            }
        }
        
        // Handle avatar upload if provided
        if ($request->hasFile('avatar')) {
            $avatar = $this->handleFileImageUpload($request, 'UserAvatar');
            $validatedData['avatar'] = $avatar['path'] ?? null;
        }
        
        // Update user profile
        $user = User::findOrFail($user->id);
        foreach ($validatedData as $key => $value) {
            $user->$key = $value;
        }
        $user->save();
        
        // Get updated user
        $updatedUser = User::find($user->id);
        
        return $this->success($updatedUser, 'Profile updated successfully');
    }
  
}

