<?php

namespace App\Http\Controllers\Traits;

use App\Models\User;
use App\Models\SubcontractorCompany;
use App\Models\EmployeeSubcontractorMeta;
use App\Models\SubcontractorEmployeeInvitation;
use App\Models\EmployeeSubcontractor;
use App\Models\Project;
use App\Models\Defect;
use App\Models\Tender;
use App\Models\RequiredDocument;
use App\Models\SubcontractorRequiredDocument;
use App\Models\InductionDocument;
use App\Models\InductionDocumentSignature;
use Carbon\Carbon;

trait SubcontractorTrait
{
    /**
     * Prepare base data for subcontractor management
     * Validates subcontractor, gets base data, and filters employees
     * 
     * @param int $subcontractorId
     * @param array $ids
     * @param string $tabFilter
     * @return array|false Returns array with base data or false on error
     */
    protected function prepareSubcontractorManagementBaseData($subcontractorId, $ids, $tabFilter)
    {
        // Validate tab filter
        $validTabs = ['projects', 'employees', 'defects', 'tenders', 'documents', 'induction'];
        if (!in_array($tabFilter, $validTabs)) {
            return false;
        }
        // Validate subcontractor exists and belongs to this customer/workspace
        $subcontractor = User::where('id', $subcontractorId)
            ->where('user_type', config('constants.user_types.subcontractor'))
            ->where('del', '0')
            ->first();
        if (!$subcontractor) {
            return false;
        }
        // Check if subcontractor belongs to this customer/workspace
        $subcontractorCompany = SubcontractorCompany::where('user_id', $subcontractorId)
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->where('del', '0')
            ->with('trade')
            ->first();
        if (!$subcontractorCompany) {
            return false;
        }
        // Get project IDs from subcontractor company
        $projectIds = $subcontractorCompany->project_ids ?? [];
        // Initialize response array
        $responseData = [
            'subcontractor' => [
                'id' => $subcontractor->id,
                'company_name' => $subcontractor->company_name ?: $subcontractor->name,
                'name' => $subcontractor->name,
                'trade' => $subcontractorCompany->trade ? $subcontractorCompany->trade->title : null,
                'trade_id' => $subcontractorCompany->trade_id,
                'email' => $subcontractor->email,
                'mobile_number' => $subcontractor->mobile_number ?? null,
                'abn' => $subcontractor->abn ?? null,
                'company_logo_url' => $subcontractor->company_logo ? url($subcontractor->company_logo) : null,
            ]
        ];
        // Get employees data (needed for multiple tabs)
        $employeeMetas = EmployeeSubcontractorMeta::where('subcontractor_id', $subcontractorId)
            ->where('active', 1)
            ->get();
        $employeeIds = $employeeMetas->pluck('emp_id')->toArray();
        // Filter employees who have accepted invitations
        $validEmployeeIds = [];
        foreach ($employeeIds as $empId) {
            $hasAcceptedInvitation = SubcontractorEmployeeInvitation::where('employee_id', $empId)
                ->where('subcontractor_id', $subcontractorId)
                ->where('customer_id', $ids['customer_id'])
                ->where('workspace_id', $ids['workspace_id'])
                ->where('invitation_status', 'accepted')
                ->exists();
            if ($hasAcceptedInvitation) {
                $validEmployeeIds[] = $empId;
            }
        }
        $totalEmployees = count($validEmployeeIds);
        // Calculate all summary fields once (regardless of tab filter)
        // Count assigned projects
        $assignedProjectsCount = 0;
        if (!empty($projectIds)) {
            $assignedProjectsCount = Project::whereIn('id', $projectIds)
                ->where('is_deleted', 0)
                ->count();
        }
        // Count defects assigned to this subcontractor
        $defectsCount = Defect::where('subcontractor_id', $subcontractorId)
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->where('del', '0')
            ->count();
        // Count awarded tenders
        $awardedTendersCount = Tender::where('status', 'awarded')
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->whereHas('participants', function ($query) use ($subcontractorId, $ids) {
                $query->where('user_id', $subcontractorId)
                    ->where('customer_id', $ids['customer_id'])
                    ->where('workspace_id', $ids['workspace_id'])
                    ->where('is_assigned', true);
            })
            ->count();
        // Check induction completion status using the induction_status column
        // This is more efficient and consistent with the new column-based approach
        $inductionComplete = $subcontractorCompany->induction_status == SubcontractorCompany::INDUCTION_COMPLETE;
        
        // Get induction documents count for summary
        $inductionDocuments = InductionDocument::where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->where('del', '0')
            ->where('is_active', true)
            ->get();
        $filteredInductionDocuments = $inductionDocuments->filter(function ($doc) {
            $roleTypes = $doc->role_types ?? [];
            return in_array('all', $roleTypes) || in_array('subcontractor', $roleTypes);
        });
        
        // Check if induction documents exist (regardless of signature status)
        $inductionExists = $filteredInductionDocuments->isNotEmpty();
        
        // If no induction documents exist for subcontractors, consider complete
        if (!$inductionExists) {
            $inductionComplete = true;
        }
        // Initialize summary with calculated values (same for all tabs)
        $summary = [
            'assigned_projects' => $assignedProjectsCount,
            'total_employees' => $totalEmployees,
            'defects' => $defectsCount,
            'awarded_tenders' => $awardedTendersCount,
            'induction_completed' => $inductionComplete,
            'induction_exists' => $inductionExists,
        ];
        return [
            'subcontractor' => $subcontractor,
            'subcontractorCompany' => $subcontractorCompany,
            'projectIds' => $projectIds,
            'responseData' => $responseData,
            'summary' => $summary,
            'employeeMetas' => $employeeMetas,
            'validEmployeeIds' => $validEmployeeIds,
        ];
    }

    /**
     * Get projects data for subcontractor management
     * 
     * @param array $baseData
     * @param int $subcontractorId
     * @param array $ids
     * @return array
     */
    protected function getSubcontractorManagementProjectsData($baseData, $subcontractorId, $ids)
    {
        $projectIds = $baseData['projectIds'];
        $validEmployeeIds = $baseData['validEmployeeIds'];
        $employeeMetas = $baseData['employeeMetas'];
        $subcontractorCompany = $baseData['subcontractorCompany'];
        $summary = $baseData['summary'];
        $responseData = $baseData['responseData'];
        // Get assigned projects with details
        $assignedProjects = [];
        if (!empty($projectIds)) {
            $projects = Project::whereIn('id', $projectIds)
                ->where('is_deleted', 0)
                ->get();
            foreach ($projects as $project) {
                $employeesLinked = 0;
                foreach ($validEmployeeIds as $empId) {
                    $meta = $employeeMetas->firstWhere('emp_id', $empId);
                    if ($meta) {
                        $empProjectIds = $meta->project_ids ?? [];
                        if (in_array($project->id, $empProjectIds)) {
                            $employeesLinked++;
                        }
                    }
                }
                $defectsAssigned = Defect::where('project_id', $project->id)
                    ->where('subcontractor_id', $subcontractorId)
                    ->where('customer_id', $ids['customer_id'])
                    ->where('workspace_id', $ids['workspace_id'])
                    ->where('del', '0')
                    ->count();
                $assignedDate = null;
                if ($subcontractorCompany->created_at) {
                    if (is_string($subcontractorCompany->created_at)) {
                        $assignedDate = Carbon::parse($subcontractorCompany->created_at)->format('Y-m-d');
                    } else {
                        $assignedDate = $subcontractorCompany->created_at->format('Y-m-d');
                    }
                }
                $assignedProjects[] = [
                    'id' => $project->id,
                    'project_name' => $project->title,
                    'assigned_date' => $assignedDate,
                    'employees_linked' => $employeesLinked,
                    'defects_assigned' => $defectsAssigned,
                ];
            }
        }
        $responseData['summary'] = $summary;
        $responseData['assigned_projects'] = $assignedProjects;
        return $responseData;
    }

    /**
     * Get employees data for subcontractor management
     * 
     * @param array $baseData
     * @param int $subcontractorId
     * @param array $ids
     * @return array
     */
    protected function getSubcontractorManagementEmployeesData($baseData, $subcontractorId, $ids)
    {
        $validEmployeeIds = $baseData['validEmployeeIds'];
        $employeeMetas = $baseData['employeeMetas'];
        $summary = $baseData['summary'];
        $responseData = $baseData['responseData'];
        // Get employee details
        $employees = [];
        if (!empty($validEmployeeIds)) {
            $employeeRecords = EmployeeSubcontractor::whereIn('id', $validEmployeeIds)->get();
            $invitations = SubcontractorEmployeeInvitation::whereIn('employee_id', $validEmployeeIds)
                ->where('subcontractor_id', $subcontractorId)
                ->where('customer_id', $ids['customer_id'])
                ->where('workspace_id', $ids['workspace_id'])
                ->where('invitation_status', 'accepted')
                ->get()
                ->groupBy('employee_id');
            // Collect all project IDs from all employee metas
            $allProjectIds = [];
            foreach ($employeeMetas as $meta) {
                if (!empty($meta->project_ids) && is_array($meta->project_ids)) {
                    $allProjectIds = array_merge($allProjectIds, $meta->project_ids);
                }
            }
            $allProjectIds = array_unique($allProjectIds);
            // Fetch all projects in one query
            $projectsMap = [];
            if (!empty($allProjectIds)) {
                $projects = Project::whereIn('id', $allProjectIds)
                    ->where('is_deleted', 0)
                    ->select('id', 'title')
                    ->get()
                    ->keyBy('id');
                $projectsMap = $projects->toArray();
            }
            foreach ($employeeRecords as $employee) {
                $employeeMeta = $employeeMetas->firstWhere('emp_id', $employee->id);
                $employeeInvitation = $invitations->get($employee->id)?->first();
                // Check induction and required documents status
                $inductionCompleted = false;
                $requiredDocumentsCompleted = false;
                if ($employeeInvitation) {
                    // Check if induction is completed (status = 1)
                    $inductionCompleted = $employeeInvitation->induction_status == SubcontractorEmployeeInvitation::INDUCTION_STATUS_COMPLETE;
                    // Check if required documents are completed (status = 1)
                    $requiredDocumentsCompleted = $employeeInvitation->required_docs_status == SubcontractorEmployeeInvitation::DOCS_STATUS_COMPLETE;
                }
                // Build projects array with IDs and titles for this employee
                $employeeProjectIds = $employeeMeta->project_ids ?? [];
                $employeeProjects = [];
                foreach ($employeeProjectIds as $projectId) {
                    if (isset($projectsMap[$projectId])) {
                        $employeeProjects[] = [
                            'id' => $projectId,
                            'title' => $projectsMap[$projectId]['title'] ?? null,
                        ];
                    } else {
                        // Include project ID even if project not found (for consistency)
                        $employeeProjects[] = [
                            'id' => $projectId,
                            'title' => null,
                        ];
                    }
                }
                $employees[] = [
                    'id' => $employee->id,
                    'first_name' => $employee->first_name,
                    'middle_name' => $employee->middle_name ?? null,
                    'last_name' => $employee->last_name,
                    'email' => $employee->email,
                    'phone' => $employee->phone ?? null,
                    'profile_image' => $employee->profile_image ? url($employee->profile_image) : null,
                    'projects' => $employeeProjects,
                    'active' => $employeeMeta->active ?? true,
                    'induction_completed' => $inductionCompleted,
                    'required_documents_completed' => $requiredDocumentsCompleted,
                ];
            }
        }
        $responseData['summary'] = $summary;
        $responseData['employees'] = $employees;
        return $responseData;
    }

    /**
     * Get defects data for subcontractor management
     * 
     * @param array $baseData
     * @param int $subcontractorId
     * @param array $ids
     * @return array
     */
    protected function getSubcontractorManagementDefectsData($baseData, $subcontractorId, $ids)
    {
        $summary = $baseData['summary'];
        $responseData = $baseData['responseData'];
        // Get all defects assigned to this subcontractor
        $defects = Defect::where('subcontractor_id', $subcontractorId)
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->where('del', '0')
            ->with(['project:id,title', 'site:id,title', 'subcontractor', 'assignedSubcontractorEmployeePersonalDetails'])
            ->orderBy('created_at', 'desc')
            ->get()
            ->map(function ($defect) {
                $dueDate = null;
                if ($defect->due_date) {
                    if (is_string($defect->due_date)) {
                        $dueDate = Carbon::parse($defect->due_date)->format('Y-m-d');
                    } else {
                        $dueDate = $defect->due_date->format('Y-m-d');
                    }
                }
                $createdAt = null;
                if ($defect->created_at) {
                    if (is_string($defect->created_at)) {
                        $createdAt = Carbon::parse($defect->created_at)->format('Y-m-d H:i:s');
                    } else {
                        $createdAt = $defect->created_at->format('Y-m-d H:i:s');
                    }
                }
                $assignedName = null;
                $assignedId = null;
                if ($defect->assignment_type === 'subcontractor') {
                    if ($defect->subcontractor) {
                        $assignedName = $defect->subcontractor->name ?: $defect->subcontractor->company_name;
                        $assignedId = $defect->subcontractor_id;
                    }
                } elseif ($defect->assignment_type === 'subcontractor_employee') {
                    if ($defect->assignedSubcontractorEmployeePersonalDetails) {
                        $assignedName = trim(
                            $defect->assignedSubcontractorEmployeePersonalDetails->first_name . ' ' .
                                ($defect->assignedSubcontractorEmployeePersonalDetails->middle_name ?? '') . ' ' .
                                $defect->assignedSubcontractorEmployeePersonalDetails->last_name
                        );
                        $assignedId = $defect->assigned_subcontractor_emp_id;
                    }
                }
                return [
                    'id' => $defect->id,
                    'title' => $defect->title,
                    'description' => $defect->description,
                    'priority' => $defect->priority,
                    'status' => $defect->status,
                    'assignment_type' => $defect->assignment_type,
                    'due_date' => $dueDate,
                    'project_id' => $defect->project_id,
                    'project_title' => $defect->project ? $defect->project->title : null,
                    'site_id' => $defect->site_id,
                    'site_title' => $defect->site ? $defect->site->title : null,
                    'assigned_id' => $assignedId,
                    'assigned_name' => $assignedName,
                    'assigned_employee_id' => $defect->assigned_subcontractor_emp_id,
                    'assigned_employee_name' => $assignedName,
                    'created_at' => $createdAt,
                    ];
            });
        $responseData['summary'] = $summary;
        $responseData['defects'] = $defects;
        return $responseData;
    }

    /**
     * Get tenders data for subcontractor management
     * 
     * @param array $baseData
     * @param int $subcontractorId
     * @param array $ids
     * @return array
     */
    protected function getSubcontractorManagementTendersData($baseData, $subcontractorId, $ids)
    {
        $summary = $baseData['summary'];
        $responseData = $baseData['responseData'];
        // Get awarded tenders
        $awardedTenders = Tender::where('status', 'awarded')
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->whereHas('participants', function ($query) use ($subcontractorId, $ids) {
                $query->where('user_id', $subcontractorId)
                    ->where('customer_id', $ids['customer_id'])
                    ->where('workspace_id', $ids['workspace_id'])
                    ->where('is_assigned', true);
            })
            ->with(['project:id,title', 'participants' => function ($query) use ($subcontractorId) {
                $query->where('user_id', $subcontractorId);
            }])
            ->orderBy('created_at', 'desc')
            ->get()
            ->map(function ($tender) {
                $participant = $tender->participants->first();
                $closingDate = null;
                if ($tender->closing_date) {
                    if (is_string($tender->closing_date)) {
                        $closingDate = Carbon::parse($tender->closing_date)->format('Y-m-d');
                    } else {
                        $closingDate = $tender->closing_date->format('Y-m-d');
                    }
                }
                $tenderIssuedDate = null;
                if ($tender->tender_issued_date) {
                    if (is_string($tender->tender_issued_date)) {
                        $tenderIssuedDate = Carbon::parse($tender->tender_issued_date)->format('Y-m-d');
                    } else {
                        $tenderIssuedDate = $tender->tender_issued_date->format('Y-m-d');
                    }
                }
                $assignedAt = null;
                if ($participant && $participant->assigned_at) {
                    if (is_string($participant->assigned_at)) {
                        $assignedAt = Carbon::parse($participant->assigned_at)->format('Y-m-d H:i:s');
                    } else {
                        $assignedAt = $participant->assigned_at->format('Y-m-d H:i:s');
                    }
                }
                return [
                    'id' => $tender->id,
                    'title' => $tender->title,
                    'reference' => $tender->reference,
                    'project_id' => $tender->project_id,
                    'project_title' => $tender->project ? $tender->project->title : null,
                    'status' => $tender->status,
                    'closing_date' => $closingDate,
                    'tender_issued_date' => $tenderIssuedDate,
                    'total_base_bid' => $tender->total_base_bid ?? null,
                    'assigned_at' => $assignedAt,
                    'created_at' => $tender->created_at ? (is_string($tender->created_at) ? Carbon::parse($tender->created_at)->format('Y-m-d H:i:s') : $tender->created_at->format('Y-m-d H:i:s')) : null,
                    ];
                });
        $responseData['summary'] = $summary;
        $responseData['awarded_tenders'] = $awardedTenders;
        return $responseData;
    }

    /**
     * Get documents data for subcontractor management
     * 
     * @param array $baseData
     * @param int $subcontractorId
     * @param array $ids
     * @return array
     */
    protected function getSubcontractorManagementDocumentsData($baseData, $subcontractorId, $ids)
    {
        $summary = $baseData['summary'];
        $responseData = $baseData['responseData'];
        // Get all required documents
        $requiredDocuments = RequiredDocument::where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->where('for_subcontractor', true)
            ->where('del', '0')
            ->with(['requiredDocumentActiveField'])
            ->get();
        $uploadedDocuments = SubcontractorRequiredDocument::where('subcontractor_id', $subcontractorId)
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->where('del', '0')
            ->get()
            ->groupBy('required_document_id');
        
        // Get system date format from BaseModel
        $baseModel = new SubcontractorRequiredDocument();
        $systemDateFormat = $baseModel->getUserDateFormat();
        
        // Map required documents with document_type
        $requiredDocsArray = $requiredDocuments->map(function ($doc) use ($uploadedDocuments, $subcontractorId, $ids, $systemDateFormat) {
            $docId = $doc->id;
            $uploaded = $uploadedDocuments->get($docId) ?? collect();
            $uploadedFieldValues = [];
            foreach ($uploaded as $uploadedDoc) {
                $uploadedFieldValues[$uploadedDoc->required_document_field_id] = [
                    'id' => $uploadedDoc->id,
                    'value' => $uploadedDoc->value,
                    'file_url' => ($uploadedDoc->value && (strpos($uploadedDoc->value, 'SubcontractorDocuments') !== false || strpos($uploadedDoc->value, 'upload') !== false))
                        ? url($uploadedDoc->value)
                        : null,
                    'updated_at' => $uploadedDoc->updated_at,
                    'approval_status' => $uploadedDoc->approval_status ?? 'pending',
                    'rejection_reason' => $uploadedDoc->rejection_reason,
                ];
            }
            $isComplete = true;

            $fields = $doc->requiredDocumentActiveField ?? collect();
            
            $requiredFields = $fields->where('field_required', true);
            $optionalFields = $fields->where('field_required', false);
            
            $hasAnySubmittedValue = false;
            
            // Detect if ANY field has a submitted value
            foreach ($fields as $field) {
                if (
                    isset($uploadedFieldValues[$field->id]) &&
                    !empty($uploadedFieldValues[$field->id]['value'])
                ) {
                    $hasAnySubmittedValue = true;
                    break;
                }
            }
            
            /**
             * CASE 1: No fields
             */
            if ($fields->isEmpty()) {
                $isComplete = true;
            }
            
            /**
             * CASE 2: Only optional fields
             * → Any ONE submitted value makes it complete
             */
            elseif ($requiredFields->isEmpty()) {
                $isComplete = $hasAnySubmittedValue;
            }
            
            /**
             * CASE 3: Required + optional fields
             * → ALL required fields must be submitted
             */
            else {
                foreach ($requiredFields as $field) {
                    if (
                        !isset($uploadedFieldValues[$field->id]) ||
                        empty($uploadedFieldValues[$field->id]['value'])
                    ) {
                        $isComplete = false;
                        break;
                    }
                }
            }
            $docApprovalStatus = null;
            if ($uploaded->isNotEmpty()) {
                $allApproved = $uploaded->every(fn($d) => $d->approval_status === 'approved');
                $anyRejected = $uploaded->contains(fn($d) => $d->approval_status === 'rejected');
                if ($allApproved) {
                    $docApprovalStatus = 'approved';
                } elseif ($anyRejected) {
                    $docApprovalStatus = 'rejected';
                } else {
                    $docApprovalStatus = 'pending';
                }
            }
            return [
                'id' => $doc->id,
                'title' => $doc->title,
                'document_type' => 'required_document',
                'is_complete' => $isComplete,
                'approval_status' => $docApprovalStatus,
                'fields' => $doc->requiredDocumentActiveField->map(function ($field) use ($uploadedFieldValues, $systemDateFormat) {
                    $fieldId = $field->id;
                    $uploadedValue = $uploadedFieldValues[$fieldId] ?? null;
                    
                    // Format date values according to system settings
                    $formattedValue = null;
                    if ($uploadedValue && $uploadedValue['value']) {
                        // If field type is date, format the stored Y-m-d date to system format
                        if ($field->field_type === 'date') {
                            try {
                                // Parse the stored date (Y-m-d format) and format it using system date format
                                $carbonDate = Carbon::createFromFormat('Y-m-d', $uploadedValue['value']);
                                $formattedValue = $carbonDate->format($systemDateFormat);
                            } catch (\Exception $e) {
                                // If parsing fails, try Carbon's parse method as fallback
                                try {
                                    $carbonDate = Carbon::parse($uploadedValue['value']);
                                    $formattedValue = $carbonDate->format($systemDateFormat);
                                } catch (\Exception $e2) {
                                    // If all parsing fails, return original value
                                    $formattedValue = $uploadedValue['value'];
                                }
                            }
                        } else {
                            // For non-date fields, return value as is
                            $formattedValue = $uploadedValue['value'];
                        }
                    }
                    
                    return [
                        'id' => $field->id,
                        'field_name' => $field->field_name,
                        'field_type' => $field->field_type,
                        'uploaded_value' => $formattedValue,
                        'uploaded_file_url' => $uploadedValue ? $uploadedValue['file_url'] : null,
                        'approval_status' => $uploadedValue ? $uploadedValue['approval_status'] : null,
                        'last_updated' => $uploadedValue ? $uploadedValue['updated_at'] : null,
                    ];
                }),
            ];
        });
        // Use only required documents (no induction documents here - they are handled separately)
        $allDocuments = $requiredDocsArray;
        // Filter uploaded/approved documents (only required documents)
        $uploadedDocs = $allDocuments->filter(function ($doc) {
            return $doc['is_complete'] === true && $doc['approval_status'] === 'approved';
        })->values();
        // Filter pending documents (only required documents)
        $pendingDocs = $allDocuments->filter(function ($doc) {
            if (!$doc['is_complete']) {
                return true;
            }
            if ($doc['approval_status'] === 'pending' || empty($doc['approval_status'])) {
                return true;
            }
            if ($doc['approval_status'] === 'rejected') {
                return true;
            }
            return false;
        })->values();
        // Summary is already calculated in base function, no need to recalculate
        $responseData['summary'] = $summary;
        $responseData['uploaded_documents'] = $uploadedDocs;
        $responseData['pending_documents'] = $pendingDocs;
        $responseData['all_documents'] = $allDocuments->values();
        return $responseData;
    }

    /**
     * Get induction data for subcontractor management
     * 
     * @param array $baseData
     * @param int $subcontractorId
     * @param array $ids
     * @return array
     */
    protected function getSubcontractorManagementInductionData($baseData, $subcontractorId, $ids)
    {
        $summary = $baseData['summary'];
        $responseData = $baseData['responseData'];
        
        // Get all active induction documents that are for subcontractors
        $inductionDocuments = InductionDocument::where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->where('del', '0')
            ->where('is_active', true)
            ->orderBy('title')
            ->orderBy('version', 'desc')
            ->get();

        // Filter documents that include 'subcontractor' in role_types or 'all'
        $filteredInductionDocuments = $inductionDocuments->filter(function ($doc) {
            $roleTypes = $doc->role_types ?? [];
            return in_array('all', $roleTypes) || in_array('subcontractor', $roleTypes);
        });

        // Get signature status for each induction document
        $inductionDocumentsWithStatus = $filteredInductionDocuments->map(function ($doc) use ($subcontractorId, $ids) {
            // Check if subcontractor has signed this document version
            $hasSigned = InductionDocumentSignature::where('induction_document_id', $doc->id)
                ->where('document_version', $doc->version)
                ->where('user_id', $subcontractorId)
                ->where('is_valid', true)
                ->where('del', '0')
                ->exists();

            // For minor updates, check if any previous version was signed
            // Minor updates do NOT require re-signing
            if (!$hasSigned && $doc->update_type === 'minor') {
                $hasSigned = SubcontractorCompany::hasSignedAnyVersionInChain(
                    $doc, 
                    $subcontractorId, 
                    $ids['customer_id'], 
                    $ids['workspace_id']
                );
            }

            // Get signature details if signed
            $signature = null;
            if ($hasSigned) {
                // First try to get signature for current version
                $signature = InductionDocumentSignature::where('induction_document_id', $doc->id)
                    ->where('document_version', $doc->version)
                    ->where('user_id', $subcontractorId)
                    ->where('is_valid', true)
                    ->where('del', '0')
                    ->first();
                    
                // If no signature for current version but minor update, get from any version in chain
                if (!$signature && $doc->update_type === 'minor') {
                    // Find root document and get signature from chain
                    $rootDocumentId = $doc->id;
                    $currentDoc = $doc;
                    while ($currentDoc && $currentDoc->parent_document_id) {
                        $rootDocumentId = $currentDoc->parent_document_id;
                        $currentDoc = InductionDocument::find($rootDocumentId);
                    }
                    
                    $documentIdsInChain = InductionDocument::where(function($q) use ($rootDocumentId) {
                            $q->where('id', $rootDocumentId)
                              ->orWhere('parent_document_id', $rootDocumentId);
                        })
                        ->where('customer_id', $ids['customer_id'])
                        ->where('workspace_id', $ids['workspace_id'])
                        ->where('del', '0')
                        ->pluck('id')
                        ->toArray();
                        
                    $signature = InductionDocumentSignature::whereIn('induction_document_id', $documentIdsInChain)
                        ->where('user_id', $subcontractorId)
                        ->where('is_valid', true)
                        ->where('del', '0')
                        ->orderBy('signed_at', 'desc')
                        ->first();
                }
            }

            return [
                'id' => $doc->id,
                'title' => $doc->title,
                'document_type' => $doc->document_type,
                'version' => $doc->version,
                'update_type' => $doc->update_type,
                'file_path' => $doc->file_path,
                'file_url' => $doc->file_path ? url($doc->file_path) : null,
                'description' => $doc->description,
                'role_types' => $doc->role_types ?? [],
                'trades' => $doc->trades->map(function ($trade) {
                    return [
                        'id' => $trade->id,
                        'title' => $trade->title,
                    ];
                })->toArray(),
                'is_signed' => $hasSigned,
                'signed_at' => $signature ? $signature->signed_at : null,
                'signed_file' => $signature ? $signature->signed_file : null,
                'signed_file_url' => $signature && $signature->signed_file ? url($signature->signed_file) : null,
                'signature_path' => $signature ? $signature->signature_path : null,
                'signature_url' => $signature && $signature->signature_path ? url($signature->signature_path) : null,
                'created_at' => $doc->created_at,
                'updated_at' => $doc->updated_at,
            ];
        })->values();

        // Calculate summary statistics
        $totalDocuments = $inductionDocumentsWithStatus->count();
        $signedCount = $inductionDocumentsWithStatus->where('is_signed', true)->count();
        $pendingCount = $inductionDocumentsWithStatus->where('is_signed', false)->count();

        // Update summary with induction-specific data
        $summary['total_induction_documents'] = $totalDocuments;
        $summary['signed_induction_documents'] = $signedCount;
        $summary['pending_induction_documents'] = $pendingCount;

        $responseData['summary'] = $summary;
        $responseData['induction_documents'] = $inductionDocumentsWithStatus;
        $responseData['total_documents'] = $totalDocuments;
        $responseData['signed_count'] = $signedCount;
        $responseData['pending_count'] = $pendingCount;

        return $responseData;
    }
}

