<?php

namespace App\Http\Controllers\Traits;

use App\Models\XeroPayslipHistory;
use App\Models\EmpCompanyDetails;
use App\Models\EmployeeAttendance;
use App\Models\RosterAssign;
use App\Models\Project;
use App\Models\ProjectSite;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

trait TimesheetTrait
{
    /**
     * Validate and fetch Xero employee data, PayTemplate, and PayrollCalendar
     * Returns array with xeroEmployee, payrollCalendar, payrollCalendarId or null on failure
     */
    protected function validateAndFetchXeroEmployeeData($xeroService, $employee, &$failedTimesheets, $customerId = null, $workspaceId = null)
    {
        // Step 1: Fetch employee data from Xero
        Log::info('Fetching employee from Xero', [
            'employee_id' => $employee->id,
            'xero_emp_id' => $employee->xero_emp_id,
            'customer_id' => $customerId,
            'workspace_id' => $workspaceId,
        ]);
        try {
            $xeroEmployeeResponse = $xeroService->makeRequest('GET', '/payroll.xro/1.0/Employees/' . $employee->xero_emp_id, [], $customerId, $workspaceId);
        } catch (\Exception $e) {
            $errorMessage = $e->getMessage();
            
            // Parse JSON error response if present
            if (preg_match('/\{.*"Status":\s*401.*\}/', $errorMessage, $matches)) {
                $errorJson = json_decode($matches[0], true);
                if ($errorJson && isset($errorJson['Detail'])) {
                    $errorMessage = $errorJson['Detail'] . ' (Status: ' . ($errorJson['Status'] ?? 401) . ')';
                } elseif ($errorJson && isset($errorJson['Title'])) {
                    $errorMessage = $errorJson['Title'] . ' (Status: ' . ($errorJson['Status'] ?? 401) . ')';
                }
            }
            
            // Check if it's an authentication error
            if (strpos($errorMessage, 'Unauthorized') !== false || strpos($errorMessage, '401') !== false || strpos($errorMessage, 'AuthorizationUnsuccessful') !== false) {
                $errorMessage = 'Xero authentication failed. The access token may be expired or invalid. Please reconnect your Xero account from settings.';
            }
            
            Log::error('Xero API request failed', [
                'employee_id' => $employee->id,
                'xero_emp_id' => $employee->xero_emp_id,
                'error' => $e->getMessage(),
                'parsed_error' => $errorMessage,
                'customer_id' => $customerId,
                'workspace_id' => $workspaceId,
            ]);
            $failedTimesheets[] = [
                'employee_id' => $employee->id,
                'employee_email' => $employee->employee_email,
                'xero_emp_id' => $employee->xero_emp_id,
                'error' => $errorMessage,
            ];
            return null;
        }
        // Extract employee data from response
        $xeroEmployee = null;
        if (isset($xeroEmployeeResponse['Employees']) && is_array($xeroEmployeeResponse['Employees']) && count($xeroEmployeeResponse['Employees']) > 0) {
            $xeroEmployee = $xeroEmployeeResponse['Employees'][0];
        } elseif (isset($xeroEmployeeResponse['EmployeeID'])) {
            $xeroEmployee = $xeroEmployeeResponse;
        }
        if (!$xeroEmployee) {
            $failedTimesheets[] = [
                'employee_id' => $employee->id,
                'employee_email' => $employee->employee_email,
                'xero_emp_id' => $employee->xero_emp_id,
                'error' => 'Employee not found in Xero',
            ];
            return null;
        }
        // Step 2: Validate PayTemplate exists
        if (!isset($xeroEmployee['PayTemplate']) || empty($xeroEmployee['PayTemplate'])) {
            $failedTimesheets[] = [
                'employee_id' => $employee->id,
                'employee_email' => $employee->employee_email,
                'xero_emp_id' => $employee->xero_emp_id,
                'error' => 'Pay Template is not configured for this employee in Xero. Please set up the Pay Template in your Xero account to sync timesheets.',
            ];
            return null;
        }
        // Step 3: Validate PayrollCalendarID exists
        if (!isset($xeroEmployee['PayrollCalendarID']) || empty($xeroEmployee['PayrollCalendarID'])) {
            $failedTimesheets[] = [
                'employee_id' => $employee->id,
                'employee_email' => $employee->employee_email,
                'xero_emp_id' => $employee->xero_emp_id,
                'error' => 'Payroll Calendar is not configured for this employee in Xero. Please set up the Payroll Calendar in your Xero account to sync timesheets.',
            ];
            return null;
        }
        // Step 4: Fetch PayrollCalendar
        $payrollCalendarId = $xeroEmployee['PayrollCalendarID'];
        Log::info('Fetching payroll calendar from Xero', [
            'payroll_calendar_id' => $payrollCalendarId,
        ]);
        try {
            $payrollCalendarResponse = $xeroService->makeRequest('GET', '/payroll.xro/1.0/PayrollCalendars/' . $payrollCalendarId, [], $customerId, $workspaceId);
        } catch (\Exception $e) {
            Log::error('Xero API request failed for PayrollCalendar', [
                'employee_id' => $employee->id,
                'xero_emp_id' => $employee->xero_emp_id,
                'payroll_calendar_id' => $payrollCalendarId,
                'error' => $e->getMessage(),
                'customer_id' => $customerId,
                'workspace_id' => $workspaceId,
            ]);
            $failedTimesheets[] = [
                'employee_id' => $employee->id,
                'employee_email' => $employee->employee_email,
                'xero_emp_id' => $employee->xero_emp_id,
                'error' => 'Xero API Error fetching PayrollCalendar: ' . $e->getMessage(),
            ];
            return null;
        }
        // Extract payroll calendar data
        $payrollCalendar = null;
        if (isset($payrollCalendarResponse['PayrollCalendars']) && is_array($payrollCalendarResponse['PayrollCalendars']) && count($payrollCalendarResponse['PayrollCalendars']) > 0) {
            $payrollCalendar = $payrollCalendarResponse['PayrollCalendars'][0];
        } elseif (isset($payrollCalendarResponse['PayrollCalendarID'])) {
            $payrollCalendar = $payrollCalendarResponse;
        }
        if (!$payrollCalendar) {
            $failedTimesheets[] = [
                'employee_id' => $employee->id,
                'employee_email' => $employee->employee_email,
                'xero_emp_id' => $employee->xero_emp_id,
                'error' => 'Payroll Calendar not found in Xero',
            ];
            return null;
        }
        // Log PayrollCalendar details for debugging
        Log::info('PayrollCalendar details', [
            'payroll_calendar_id' => $payrollCalendarId,
            'calendar_type' => $payrollCalendar['CalendarType'] ?? null,
            'start_date' => $payrollCalendar['StartDate'] ?? null,
            'payment_date' => $payrollCalendar['PaymentDate'] ?? null,
            'name' => $payrollCalendar['Name'] ?? null,
            'full_response' => $payrollCalendar,
        ]);
        return [
            'xeroEmployee' => $xeroEmployee,
            'payrollCalendar' => $payrollCalendar,
            'payrollCalendarId' => $payrollCalendarId,
        ];
    }

    /**
     * Calculate pay period dates from PayrollCalendar or existing Xero timesheets
     * Returns array with startDate, endDate, startDateCarbon, endDateCarbon, useXeroDateStrings, periodDates or null on failure
     */
    protected function calculatePayPeriodDates($xeroService, $employee, $payrollCalendar, $payrollCalendarId, &$failedTimesheets)
    {
        // Step 5: Try to get valid period from existing Xero timesheets first
        // This ensures we use periods that Xero actually accepts
        // BUT: Validate that the period matches the calendar type
        $calendarType = strtoupper($payrollCalendar['CalendarType'] ?? '');
        $periodDates = $xeroService->getValidPeriodFromXeroTimesheets($employee->xero_emp_id, $payrollCalendarId);
        // Validate that the found period matches the calendar type
        if ($periodDates && $calendarType) {
            $foundStart = Carbon::parse($periodDates['start_date']);
            $foundEnd = Carbon::parse($periodDates['end_date']);
            $periodLength = $foundStart->diffInDays($foundEnd) + 1;
            $isValidPeriod = false;
            if ($calendarType === 'MONTHLY') {
                // For monthly, period should be approximately 28-31 days (full month)
                $isValidPeriod = ($periodLength >= 28 && $periodLength <= 31);
            } elseif ($calendarType === 'WEEKLY') {
                // For weekly, period should be 7 days
                $isValidPeriod = ($periodLength >= 6 && $periodLength <= 7);
            } elseif (in_array($calendarType, ['FORTNIGHTLY', 'BIWEEKLY'])) {
                // For fortnightly, period should be 14 days
                $isValidPeriod = ($periodLength >= 13 && $periodLength <= 14);
            } else {
                // For unknown types, accept the period
                $isValidPeriod = true;
            }
            
            if (!$isValidPeriod) {
                Log::warning('Found period from existing timesheet does not match calendar type, recalculating', [
                    'calendar_type' => $calendarType,
                    'found_period_length' => $periodLength,
                    'found_start' => $periodDates['start_date'],
                    'found_end' => $periodDates['end_date'],
                ]);
                $periodDates = null; // Force recalculation
            }
        }
        
        // If we couldn't get from existing timesheets or period doesn't match, calculate from PayrollCalendar
        if (!$periodDates) {
            $periodDates = $xeroService->calculateCurrentPeriodDates($payrollCalendar);
            if (!$periodDates) {
                $failedTimesheets[] = [
                    'employee_id' => $employee->id,
                    'employee_email' => $employee->employee_email,
                    'xero_emp_id' => $employee->xero_emp_id,
                    'error' => 'Unable to calculate current pay period dates from Payroll Calendar.',
                ];
                return null;
            }
        }
        $startDate = $periodDates['start_date'];
        $endDate = $periodDates['end_date'];
        $startDateCarbon = Carbon::parse($startDate);
        $endDateCarbon = Carbon::parse($endDate);
        
        // If we have original Xero date strings from existing timesheet, use them directly
        // This preserves the exact format Xero expects
        $useXeroDateStrings = isset($periodDates['xero_start_date']) && isset($periodDates['xero_end_date']);
        
        if ($useXeroDateStrings) {
            Log::info('Using period from existing Xero timesheet with original date strings', [
                'start_date' => $startDate,
                'end_date' => $endDate,
                'xero_start_date' => $periodDates['xero_start_date'],
                'xero_end_date' => $periodDates['xero_end_date'],
            ]);
        }

        Log::info('Calculated current period dates from PayrollCalendar', [
            'employee_id' => $employee->id,
            'calendar_type' => $payrollCalendar['CalendarType'] ?? 'UNKNOWN',
            'start_date' => $startDate,
            'end_date' => $endDate,
        ]);

        return [
            'startDate' => $startDate,
            'endDate' => $endDate,
            'startDateCarbon' => $startDateCarbon,
            'endDateCarbon' => $endDateCarbon,
            'useXeroDateStrings' => $useXeroDateStrings,
            'periodDates' => $periodDates,
        ];
    }

    /**
     * Prepare timesheet data: get EarningsRateID, calculate daily hours, format dates, check existing timesheet
     * Returns array with earningsRateId, dailyHours, totalHours, startDateFormatted, endDateFormatted, 
     * startDateFormattedDb, endDateFormattedDb, existingTimesheet, isUpdate, xeroTimesheetId or null on failure
     */
    protected function prepareTimesheetData($xeroService, $employee, $xeroEmployee, $startDateCarbon, $endDateCarbon, $periodDates, $useXeroDateStrings, $ids, &$failedTimesheets)
    {
        // Step 6: Get EarningsRateID from PayTemplate
        $earningsRateId = null;
        if (isset($xeroEmployee['PayTemplate']['EarningsLines']) && is_array($xeroEmployee['PayTemplate']['EarningsLines']) && count($xeroEmployee['PayTemplate']['EarningsLines']) > 0) {
            $earningsRateId = $xeroEmployee['PayTemplate']['EarningsLines'][0]['EarningsRateID'] ?? null;
        }
        if (!$earningsRateId) {
            $failedTimesheets[] = [
                'employee_id' => $employee->id,
                'employee_email' => $employee->employee_email,
                'xero_emp_id' => $employee->xero_emp_id,
                'start_date' => $periodDates['start_date'],
                'end_date' => $periodDates['end_date'],
                'error' => 'Earnings Rate is not configured in the Pay Template. Please set up Earnings Rate in your Xero account.',
            ];
            return null;
        }

        // Step 7: Calculate daily hours from attendance records
        $dailyHours = $xeroService->getDailyHoursFromAttendance($employee->id, $startDateCarbon, $endDateCarbon, $ids['customer_id'], $ids['workspace_id']);
        
        // Check if dailyHours is empty or all values are zero (no attendance data)
        $totalHours = (float) array_sum($dailyHours);
        if (empty($dailyHours) || $totalHours <= 0) {
            $failedTimesheets[] = [
                'employee_id' => $employee->id,
                'employee_email' => $employee->employee_email,
                'xero_emp_id' => $employee->xero_emp_id,
                'start_date' => $periodDates['start_date'],
                'end_date' => $periodDates['end_date'],
                'error' => 'No attendance records found for the current pay period. Please ensure the employee has attendance data for the period from ' . $periodDates['start_date'] . ' to ' . $periodDates['end_date'] . '.',
            ];
            return null;
        }
    
        // Step 8: Format dates for Xero (Xero date format: /Date(timestamp+timezone)/)
        // If we have original Xero date strings from existing timesheet, use them directly
        // Otherwise, format from Carbon dates
        if ($useXeroDateStrings) {
            $startDateFormatted = $periodDates['xero_start_date'];
            $endDateFormatted = $periodDates['xero_end_date'];
            
            Log::info('Using original Xero date strings from existing timesheet', [
                'start_date_formatted' => $startDateFormatted,
                'end_date_formatted' => $endDateFormatted,
            ]);
        } else {
            // Use start of day in UTC for both dates (Xero examples show both use 00:00:00)
            // Create dates directly in UTC without timezone conversion
            // Parse the date string and create UTC date at midnight
            $startDateUtc = Carbon::create(
                $startDateCarbon->year,
                $startDateCarbon->month,
                $startDateCarbon->day,
                0, 0, 0,
                'UTC'
            );
            $endDateUtc = Carbon::create(
                $endDateCarbon->year,
                $endDateCarbon->month,
                $endDateCarbon->day,
                0, 0, 0,
                'UTC'
            );
            
            $startDateTimestamp = $startDateUtc->timestamp * 1000; // Convert to milliseconds
            $endDateTimestamp = $endDateUtc->timestamp * 1000;
            $startDateFormatted = '/Date(' . $startDateTimestamp . '+0000)/';
            $endDateFormatted = '/Date(' . $endDateTimestamp . '+0000)/';
            
            Log::info('Formatted dates for Xero API', [
                'start_date_carbon' => $startDateCarbon->format('Y-m-d'),
                'end_date_carbon' => $endDateCarbon->format('Y-m-d'),
                'start_date_utc' => $startDateUtc->format('Y-m-d H:i:s T'),
                'end_date_utc' => $endDateUtc->format('Y-m-d H:i:s T'),
                'start_date_formatted' => $startDateFormatted,
                'end_date_formatted' => $endDateFormatted,
                'start_timestamp' => $startDateTimestamp,
                'end_timestamp' => $endDateTimestamp,
            ]);
        }

        $totalHours = array_sum($dailyHours);
        $startDateFormattedDb = $startDateCarbon->format('Y-m-d');
        $endDateFormattedDb = $endDateCarbon->format('Y-m-d');

        // Step 9: Check if timesheet already exists in xero_payslip_history
        $existingTimesheet = XeroPayslipHistory::where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->where('employee_id', $employee->id)
            ->where('xero_emp_id', $employee->xero_emp_id)
            ->where('start_date', $startDateFormattedDb)
            ->where('end_date', $endDateFormattedDb)
            ->first();

        $isUpdate = false;
        $xeroTimesheetId = null;
        if ($existingTimesheet) {
            // Try to get xero_timesheet_id from database field first
            if (!empty($existingTimesheet->xero_timesheet_id)) {
                $xeroTimesheetId = $existingTimesheet->xero_timesheet_id;
                $isUpdate = true;
                Log::info('Found existing timesheet ID in database', [
                    'employee_id' => $employee->id,
                    'xero_emp_id' => $employee->xero_emp_id,
                    'xero_timesheet_id' => $xeroTimesheetId,
                    'start_date' => $startDateFormattedDb,
                    'end_date' => $endDateFormattedDb,
                ]);
            } elseif (!empty($existingTimesheet->xero_response)) {
                // Try to extract TimesheetID from xero_response JSON
                try {
                    $xeroResponseData = json_decode($existingTimesheet->xero_response, true);
                    if (is_array($xeroResponseData)) {
                        // Check in Timesheets array
                        if (isset($xeroResponseData['Timesheets']) && is_array($xeroResponseData['Timesheets']) && count($xeroResponseData['Timesheets']) > 0) {
                            $xeroTimesheetId = $xeroResponseData['Timesheets'][0]['TimesheetID'] ?? null;
                        } elseif (isset($xeroResponseData['TimesheetID'])) {
                            $xeroTimesheetId = $xeroResponseData['TimesheetID'];
                        }
                        
                        if ($xeroTimesheetId) {
                            $isUpdate = true;
                            // Update the database record with the extracted ID
                            $existingTimesheet->update(['xero_timesheet_id' => $xeroTimesheetId]);
                            Log::info('Extracted and saved TimesheetID from xero_response', [
                                'employee_id' => $employee->id,
                                'xero_emp_id' => $employee->xero_emp_id,
                                'xero_timesheet_id' => $xeroTimesheetId,
                                'start_date' => $startDateFormattedDb,
                                'end_date' => $endDateFormattedDb,
                            ]);
                        }
                    }
                } catch (\Exception $e) {
                    Log::warning('Failed to extract TimesheetID from xero_response', [
                        'employee_id' => $employee->id,
                        'error' => $e->getMessage(),
                    ]);
                }
            }
            
            if ($isUpdate) {
                Log::info('Found existing timesheet in history, will update', [
                    'employee_id' => $employee->id,
                    'xero_emp_id' => $employee->xero_emp_id,
                    'xero_timesheet_id' => $xeroTimesheetId,
                    'start_date' => $startDateFormattedDb,
                    'end_date' => $endDateFormattedDb,
                ]);
            } else {
                Log::info('Found existing timesheet record but no TimesheetID available', [
                    'employee_id' => $employee->id,
                    'xero_emp_id' => $employee->xero_emp_id,
                    'start_date' => $startDateFormattedDb,
                    'end_date' => $endDateFormattedDb,
                    'has_xero_response' => !empty($existingTimesheet->xero_response),
                ]);
            }
        }

        return [
            'earningsRateId' => $earningsRateId,
            'dailyHours' => $dailyHours,
            'totalHours' => $totalHours,
            'startDateFormatted' => $startDateFormatted,
            'endDateFormatted' => $endDateFormatted,
            'startDateFormattedDb' => $startDateFormattedDb,
            'endDateFormattedDb' => $endDateFormattedDb,
            'existingTimesheet' => $existingTimesheet,
            'isUpdate' => $isUpdate,
            'xeroTimesheetId' => $xeroTimesheetId,
        ];
    }

    /**
     * Sync timesheet to Xero with retry logic for period errors and "already exists" errors
     * Returns array with response, xeroTimesheetId, isUpdate or throws exception
     */
    protected function syncTimesheetToXeroWithRetry($xeroService, $employee, $timesheetPayload, $isUpdate, &$xeroTimesheetId, $existingTimesheet, $startDateFormattedDb, $endDateFormattedDb, $payrollCalendarId, $ids)
    {
        // Step 10: Prepare timesheet payload with TimesheetLines
        // If updating, include TimesheetID in payload
        if ($isUpdate) {
            $timesheetPayload['TimesheetID'] = $xeroTimesheetId;
        }

        Log::info($isUpdate ? 'Updating timesheet in Xero' : 'Creating timesheet in Xero', [
            'employee_id' => $employee->id,
            'xero_emp_id' => $employee->xero_emp_id,
            'xero_timesheet_id' => $xeroTimesheetId,
            'start_date' => $timesheetPayload['StartDate'],
            'end_date' => $timesheetPayload['EndDate'],
            'daily_hours' => $timesheetPayload['TimesheetLines'][0]['NumberOfUnits'],
            'total_hours' => array_sum($timesheetPayload['TimesheetLines'][0]['NumberOfUnits']),
            'is_update' => $isUpdate,
            'payload' => $timesheetPayload,
        ]);

        // Step 11: Make API request to Xero Timesheet endpoint
        // Xero uses POST for both create and update (when TimesheetID is included)
        $response = null;
        $periodRetried = false;
        
        try {
            $response = $xeroService->makeRequest('POST', '/payroll.xro/1.0/Timesheets', [
                'json' => [$timesheetPayload]
            ]);

            // Extract Xero timesheet ID from response if available (for new timesheets)
            if (!$isUpdate) {
                if (isset($response['Timesheets']) && is_array($response['Timesheets']) && count($response['Timesheets']) > 0) {
                    $xeroTimesheetId = $response['Timesheets'][0]['TimesheetID'] ?? null;
                } elseif (isset($response['TimesheetID'])) {
                    $xeroTimesheetId = $response['TimesheetID'];
                }
            }
        } catch (\Exception $apiException) {
            $apiErrorMessage = $apiException->getMessage();
            
            // Handle "period doesn't correspond" error - try to fetch correct period from Xero
            if (strpos($apiErrorMessage, "doesn't correspond with a pay period") !== false || 
                strpos($apiErrorMessage, 'pay period') !== false) {
                
                Log::warning('Period validation error, attempting to fetch correct period from Xero', [
                    'employee_id' => $employee->id,
                    'xero_emp_id' => $employee->xero_emp_id,
                    'calculated_start' => $startDateFormattedDb,
                    'calculated_end' => $endDateFormattedDb,
                ]);
                
                // Try to get the correct period from existing Xero timesheets
                $correctPeriod = $xeroService->getValidPeriodFromXeroTimesheets($employee->xero_emp_id, $payrollCalendarId);
                
                if ($correctPeriod && 
                    ($correctPeriod['start_date'] != $startDateFormattedDb || 
                     $correctPeriod['end_date'] != $endDateFormattedDb)) {
                    
                    Log::info('Found different period from Xero, retrying with correct period', [
                        'old_period' => ['start' => $startDateFormattedDb, 'end' => $endDateFormattedDb],
                        'new_period' => $correctPeriod,
                    ]);
                    
                    // Recalculate daily hours for the correct period
                    $correctStartCarbon = Carbon::parse($correctPeriod['start_date']);
                    $correctEndCarbon = Carbon::parse($correctPeriod['end_date']);
                    $correctDailyHours = $xeroService->getDailyHoursFromAttendance($employee->id, $correctStartCarbon, $correctEndCarbon, $ids['customer_id'], $ids['workspace_id']);
                    
                    if (!empty($correctDailyHours) && array_sum($correctDailyHours) > 0) {
                        // Update dates and hours
                        $startDateCarbon = $correctStartCarbon;
                        $endDateCarbon = $correctEndCarbon;
                        $dailyHours = $correctDailyHours;
                        $startDateFormattedDb = $correctPeriod['start_date'];
                        $endDateFormattedDb = $correctPeriod['end_date'];
                        $totalHours = array_sum($dailyHours);
                        
                        // Reformat dates for Xero
                        $startDateUtc = $startDateCarbon->copy()->startOfDay()->setTimezone('UTC');
                        $endDateUtc = $endDateCarbon->copy()->startOfDay()->setTimezone('UTC');
                        $startDateTimestamp = $startDateUtc->timestamp * 1000;
                        $endDateTimestamp = $endDateUtc->timestamp * 1000;
                        $startDateFormatted = '/Date(' . $startDateTimestamp . '+0000)/';
                        $endDateFormatted = '/Date(' . $endDateTimestamp . '+0000)/';
                        
                        // Update payload
                        $timesheetPayload['StartDate'] = $startDateFormatted;
                        $timesheetPayload['EndDate'] = $endDateFormatted;
                        $timesheetPayload['TimesheetLines'][0]['NumberOfUnits'] = array_values($dailyHours);
                        
                        // Retry the request
                        try {
                            $response = $xeroService->makeRequest('POST', '/payroll.xro/1.0/Timesheets', [
                                'json' => [$timesheetPayload]
                            ]);
                            
                            $periodRetried = true;
                            
                            // Extract Xero timesheet ID from response
                            if (!$isUpdate) {
                                if (isset($response['Timesheets']) && is_array($response['Timesheets']) && count($response['Timesheets']) > 0) {
                                    $xeroTimesheetId = $response['Timesheets'][0]['TimesheetID'] ?? null;
                                } elseif (isset($response['TimesheetID'])) {
                                    $xeroTimesheetId = $response['TimesheetID'];
                                }
                            }
                            
                            Log::info('Successfully synced timesheet after period correction', [
                                'correct_period' => $correctPeriod,
                                'xero_timesheet_id' => $xeroTimesheetId,
                            ]);
                        } catch (\Exception $retryException) {
                            // If retry also fails, throw the original error
                            throw $apiException;
                        }
                    } else {
                        // No hours for the correct period, throw original error
                        throw $apiException;
                    }
                } else {
                    // Couldn't find correct period, throw original error
                    throw $apiException;
                }
            }
            // Handle "already exists" error - try to fetch existing timesheet ID
            elseif (strpos($apiErrorMessage, 'already exists') !== false || 
                strpos($apiErrorMessage, 'provide the timesheet ID') !== false) {
                
                Log::warning('Timesheet already exists in Xero, attempting to fetch existing timesheet', [
                    'employee_id' => $employee->id,
                    'xero_emp_id' => $employee->xero_emp_id,
                    'start_date' => $startDateFormattedDb,
                    'end_date' => $endDateFormattedDb,
                ]);

                // Try to get existing timesheet ID from our database
                if (!$xeroTimesheetId && $existingTimesheet) {
                    // First try the xero_timesheet_id field
                    if (!empty($existingTimesheet->xero_timesheet_id)) {
                        $xeroTimesheetId = $existingTimesheet->xero_timesheet_id;
                        Log::info('Using existing timesheet ID from database field', [
                            'xero_timesheet_id' => $xeroTimesheetId,
                        ]);
                    } elseif (!empty($existingTimesheet->xero_response)) {
                        // Try to extract from xero_response
                        try {
                            $xeroResponseData = json_decode($existingTimesheet->xero_response, true);
                            if (is_array($xeroResponseData)) {
                                if (isset($xeroResponseData['Timesheets']) && is_array($xeroResponseData['Timesheets']) && count($xeroResponseData['Timesheets']) > 0) {
                                    $xeroTimesheetId = $xeroResponseData['Timesheets'][0]['TimesheetID'] ?? null;
                                } elseif (isset($xeroResponseData['TimesheetID'])) {
                                    $xeroTimesheetId = $xeroResponseData['TimesheetID'];
                                }
                                
                                if ($xeroTimesheetId) {
                                    // Update the database record with the extracted ID
                                    $existingTimesheet->update(['xero_timesheet_id' => $xeroTimesheetId]);
                                    Log::info('Extracted TimesheetID from xero_response during error handling', [
                                        'xero_timesheet_id' => $xeroTimesheetId,
                                    ]);
                                }
                            }
                        } catch (\Exception $e) {
                            Log::warning('Failed to extract TimesheetID from xero_response in error handler', [
                                'error' => $e->getMessage(),
                            ]);
                        }
                    }
                }

                // If we have the timesheet ID, retry with update
                if ($xeroTimesheetId) {
                    $timesheetPayload['TimesheetID'] = $xeroTimesheetId;
                    $isUpdate = true;
                    
                    Log::info('Retrying with TimesheetID to update existing timesheet', [
                        'xero_timesheet_id' => $xeroTimesheetId,
                    ]);

                    try {
                        $response = $xeroService->makeRequest('POST', '/payroll.xro/1.0/Timesheets', [
                            'json' => [$timesheetPayload]
                        ]);
                        
                        Log::info('Successfully updated existing timesheet after retry', [
                            'xero_timesheet_id' => $xeroTimesheetId,
                        ]);
                    } catch (\Exception $retryException) {
                        Log::error('Failed to update timesheet even with TimesheetID', [
                            'xero_timesheet_id' => $xeroTimesheetId,
                            'error' => $retryException->getMessage(),
                        ]);
                        throw $retryException;
                    }
                } else {
                    // If we still don't have the ID, provide a helpful error message
                    $errorMessage = "Timesheet already exists in Xero for this period, but we couldn't find the Timesheet ID. " .
                                  "Please check the xero_payslip_history table or manually sync the timesheet in Xero.";
                    Log::error('Cannot update timesheet - TimesheetID not found', [
                        'employee_id' => $employee->id,
                        'xero_emp_id' => $employee->xero_emp_id,
                        'start_date' => $startDateFormattedDb,
                        'end_date' => $endDateFormattedDb,
                        'has_existing_record' => !is_null($existingTimesheet),
                    ]);
                    throw new \Exception($errorMessage);
                }
            } else {
                // For other errors, re-throw
                throw $apiException;
            }
        }

        return [
            'response' => $response,
            'xeroTimesheetId' => $xeroTimesheetId,
            'isUpdate' => $isUpdate,
        ];
    }

    /**
     * Save or update timesheet history in xero_payslip_history table
     */
    protected function saveTimesheetHistory($isUpdate, $existingTimesheet, $totalHours, $response, $xeroTimesheetId, $ids, $employee, $startDateFormattedDb, $endDateFormattedDb, $empType = 'internal', $subcontractorId = null)
    {
        // Step 12: Save or update xero_payslip_history table
        if ($isUpdate && $existingTimesheet) {
            // Update existing record
            $existingTimesheet->update([
                'hours' => $totalHours,
                'status' => 'DRAFT',
                'xero_response' => json_encode($response),
                'xero_timesheet_id' => $xeroTimesheetId, // In case it changed
                'emp_type' => $empType,
                'subcontractor_id' => $subcontractorId,
            ]);
            Log::info('Updated existing timesheet in history', [
                'history_id' => $existingTimesheet->id,
                'xero_timesheet_id' => $xeroTimesheetId,
                'emp_type' => $empType,
                'subcontractor_id' => $subcontractorId,
            ]);
        } else {
            // Create new record or update if we found the ID during error handling
            if ($existingTimesheet) {
                // Update existing record that didn't have xero_timesheet_id before
                $existingTimesheet->update([
                    'hours' => $totalHours,
                    'status' => 'DRAFT',
                    'xero_response' => json_encode($response),
                    'xero_timesheet_id' => $xeroTimesheetId,
                    'emp_type' => $empType,
                    'subcontractor_id' => $subcontractorId,
                ]);
                Log::info('Updated existing timesheet record with new Xero ID', [
                    'history_id' => $existingTimesheet->id,
                    'xero_timesheet_id' => $xeroTimesheetId,
                    'emp_type' => $empType,
                    'subcontractor_id' => $subcontractorId,
                ]);
            } else {
                // Create new record
                XeroPayslipHistory::create([
                    'customer_id' => $ids['customer_id'],
                    'workspace_id' => $ids['workspace_id'],
                    'employee_id' => $employee->id,
                    'emp_type' => $empType,
                    'subcontractor_id' => $subcontractorId,
                    'xero_emp_id' => $employee->xero_emp_id,
                    'start_date' => $startDateFormattedDb,
                    'end_date' => $endDateFormattedDb,
                    'hours' => $totalHours,
                    'status' => 'DRAFT',
                    'xero_response' => json_encode($response),
                    'xero_timesheet_id' => $xeroTimesheetId,
                ]);
                Log::info('Created new timesheet in history', [
                    'xero_timesheet_id' => $xeroTimesheetId,
                    'emp_type' => $empType,
                    'subcontractor_id' => $subcontractorId,
                ]);
            }
        }
    }

    /**
     * Chunk 1: Validate request and parse date filters
     * Returns array with fromDate, toDate, userDateFormat or null on validation failure
     */
    protected function validateAndParseTimesheetDates($request)
    {
        $validator = Validator::make($request->all(), [
            'from_date' => 'nullable|date', 
            'to_date' => 'nullable|date|after_or_equal:from_date',
            'employee_id' => 'nullable|integer|exists:emp_company_details,id', 
            'site_id' => 'nullable|integer|exists:sites,id', 
            'project_id' => 'nullable|integer|exists:projects,id',
            'subcontractor_id' => 'nullable|integer|exists:users,id', 
        ]);
        
        if ($validator->fails()) {
            return ['error' => $validator];
        }

        // Get user's preferred date format for response formatting
        $attendanceModel = new EmployeeAttendance();
        $userDateFormat = $attendanceModel->getUserDateFormat();

        if ($request->filled('from_date')) {
            $fromDate = Carbon::parse($request->from_date)->format('Y-m-d');
        } else {
            $fromDate = Carbon::now()->startOfMonth()->format('Y-m-d');
        }

        if ($request->filled('to_date')) {
            $toDate = Carbon::parse($request->to_date)->format('Y-m-d');
        } else {
            $toDate = Carbon::now()->endOfMonth()->format('Y-m-d');
        }

        return [
            'fromDate' => $fromDate,
            'toDate' => $toDate,
            'userDateFormat' => $userDateFormat,
        ];
    }

    /**
     * Chunk 2: Get subcontractor employees timesheets
     * Returns array with timesheets or null if no employees found
     */
    protected function getSubcontractorTimesheets($request, $ids)
    {
        $subcontractorEmployees = $this->subcontractorEmployeesGetter(
            $request->subcontractor_id, 
            false, 
            $ids['customer_id'], 
            true
        );

        if ($subcontractorEmployees->isEmpty()) {
            return null;
        }

        $timesheets = [];
        foreach ($subcontractorEmployees as $subEmp) {
            $employeeName = trim(($subEmp->first_name ?? '') . ' ' . 
                                ($subEmp->middle_name ?? '') . ' ' . 
                                ($subEmp->last_name ?? ''));
            if (empty(trim($employeeName))) {
                $employeeName = 'Unknown';
            }
            
            $timesheets[] = [
                'employee_id' => $subEmp->id,
                'employee_xero_id' => $subEmp->xero_emp_id ?? null,
                'employee_email' => $subEmp->email ?? '',
                'employee_name' => $employeeName,
                'employee_image' => $subEmp->profile_image ?? null,
                'attendance_count' => 0,
                'total_working_hours' => 0.0,
                'attendances' => [],
                'subcontractor_id' => $request->subcontractor_id,
            ];
        }

        return $timesheets;
    }

    /**
     * Chunk 3: Fetch internal employees data (employees, attendances, projects, roster assigns, project sites)
     * Returns array with employees, allAttendances, projects, rosterAssigns, projectSitesMap, projectIdsFromProjectSites
     */
    protected function fetchInternalEmployeesData($request, $ids, $fromDate, $toDate)
    {
        // Get all internal employees (user_type = 0) - regular employees from EmpCompanyDetails
        $employeesQuery = EmpCompanyDetails::withoutGlobalScope(\App\Scopes\NotDeletedScope::class)
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->where('user_type', 0) // Internal employees only
            ->where('del', '0');
        
        if ($request->filled('employee_id')) {
            $employeesQuery->where('id', $request->employee_id);
        }
        
        $employees = $employeesQuery->with('empPersonalDetails')->get();
        
        if ($employees->isEmpty()) {
            return null;
        }

        $employeeIds = $employees->pluck('id')->toArray();

        // Get project_ids from project_sites table when site_id filter is provided
        $projectIdsFromProjectSites = [];
        if ($request->filled('site_id')) {
            $site = \App\Models\Sites::where('id', $request->site_id)
                ->where('customer_id', $ids['customer_id'])
                ->where('workspace_id', $ids['workspace_id'])
                ->first();   
            if ($site) {
                $projectSites = ProjectSite::where('site_id', $request->site_id)
                    ->pluck('project_id')
                    ->filter()
                    ->unique()
                    ->toArray();
                $projectIdsFromProjectSites = $projectSites;
            }
        }

        // Fetch attendances for internal employees (subcontractor_id must be null)
        $attendanceQuery = EmployeeAttendance::whereIn('employee_id', $employeeIds)
            ->whereNull('subcontractor_id') // Only get attendances for internal employees
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->whereBetween('date', [$fromDate, $toDate]);

        if ($request->filled('site_id')) {
            $attendanceQuery->where('site_id', $request->site_id);
        }

        $allAttendancesCollection = $attendanceQuery->with([
            'sites' => function($query) {
                $query->select('id', 'title', 'project_id');
            },
            'sites.project' => function($query) use ($ids) {
                $query->where('customer_id', $ids['customer_id'])
                      ->where('workspace_id', $ids['workspace_id'])
                      ->select('id', 'title');
            },
            'empPersonalDetails' => function($query) {
                $query->select('emp_id', 'first_name', 'middle_name', 'last_name', 'image');
            },
            'breaks' => function($query) {
                $query->select('id', 'emp_attendance_id', 'break_in', 'break_out', 'date')
                      ->orderBy('break_in', 'asc');
            }
        ])
        ->orderBy('date', 'desc')
        ->get();

        if ($request->filled('site_id') && $allAttendancesCollection->isEmpty()) {
            return ['empty' => true];
        }

        // Get project_ids from project_sites for the filtered site_id
        $siteProjectIds = $allAttendancesCollection->pluck('sites.project_id')
            ->filter()
            ->unique()
            ->toArray();

        // Merge project_ids from project_sites table when site_id filter is provided
        if (!empty($projectIdsFromProjectSites)) {
            $siteProjectIds = array_unique(array_merge($siteProjectIds, $projectIdsFromProjectSites));
        }

        // Group by employee_id ensuring keys are integers
        $allAttendances = $allAttendancesCollection->groupBy(function($item) {
            return (int)$item->employee_id;
        });

        // Fetch roster assigns for internal employees (subcontractor_id must be null)
        $rosterAssigns = RosterAssign::whereIn('assign_to', $employeeIds)
            ->whereNull('subcontractor_id') // Only get roster assigns for internal employees
            ->whereBetween('schedule_date', [$fromDate, $toDate])
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->get()
            ->groupBy(function($item) {
                $scheduleDate = $item->schedule_date instanceof Carbon 
                    ? $item->schedule_date->format('Y-m-d') 
                    : Carbon::parse($item->schedule_date)->format('Y-m-d');
                return $item->assign_to . '_' . $scheduleDate;
            });

        $rosterProjectIds = $rosterAssigns->pluck('project_id')->filter()->unique()->toArray();
        $allProjectIds = array_unique(array_merge($rosterProjectIds, $siteProjectIds));

        // Fetch projects
        $projects = [];
        if (!empty($allProjectIds)) {
            $projects = Project::whereIn('id', $allProjectIds)
                ->where('customer_id', $ids['customer_id'])
                ->where('workspace_id', $ids['workspace_id'])
                ->select('id', 'title')
                ->get()
                ->keyBy('id');
        }

        // Get project_sites mapping for site_id to project_id lookup
        $projectSitesMap = [];
        if ($request->filled('site_id')) {
            $site = \App\Models\Sites::where('id', $request->site_id)
                ->where('customer_id', $ids['customer_id'])
                ->where('workspace_id', $ids['workspace_id'])
                ->first();
            if ($site) {
                $projectSitesRecords = ProjectSite::where('site_id', $request->site_id)->get();
                foreach ($projectSitesRecords as $ps) {
                    if (!isset($projectSitesMap[$ps->site_id])) {
                        $projectSitesMap[$ps->site_id] = [];
                    }
                    $projectSitesMap[$ps->site_id][] = $ps->project_id;
                }
            }
        } else {
            // If no site_id filter, get all project_sites for sites in the attendance data
            $siteIds = $allAttendancesCollection->pluck('site_id')->filter()->unique()->toArray();
            if (!empty($siteIds)) {
                $projectSitesRecords = ProjectSite::whereIn('site_id', $siteIds)->get();
                foreach ($projectSitesRecords as $ps) {
                    if (!isset($projectSitesMap[$ps->site_id])) {
                        $projectSitesMap[$ps->site_id] = [];
                    }
                    $projectSitesMap[$ps->site_id][] = $ps->project_id;
                }
            }
        }

        return [
            'employees' => $employees,
            'allAttendances' => $allAttendances,
            'projects' => $projects,
            'rosterAssigns' => $rosterAssigns,
            'projectSitesMap' => $projectSitesMap,
            'projectIdsFromProjectSites' => $projectIdsFromProjectSites,
        ];
    }

    /**
     * Chunk 4: Build timesheet data for internal employees
     * Returns array of timesheets with attendance data
     */
    protected function buildTimesheetData($request, $ids, $employees, $allAttendances, $projects, $rosterAssigns, $projectSitesMap, $userDateFormat)
    {
        $timesheets = [];
        
        foreach ($employees as $employee) {
            // Ensure we use integer key for lookup
            $employeeId = (int)$employee->id;
            $attendances = $allAttendances->get($employeeId, collect());
            $attendanceData = [];
            
            foreach ($attendances as $attendance) {
                // Format attendance date properly (handle both Carbon and string)
                $attendanceDate = $attendance->date instanceof Carbon 
                    ? $attendance->date->format('Y-m-d') 
                    : Carbon::parse($attendance->date)->format('Y-m-d');
                
                // Find matching roster assign by employee_id and date
                $rosterKey = (int)$employee->id . '_' . $attendanceDate;
                $rosterAssign = $rosterAssigns->get($rosterKey)?->first();
                
                // Get project information - prioritize project_sites table, then site's project, then roster_assign
                $project = null;
                $projectId = null;
                
                // First, check project_sites table (highest priority)
                if ($attendance->sites && $attendance->sites->id) {
                    $siteId = $attendance->sites->id;
                    if (isset($projectSitesMap[$siteId]) && !empty($projectSitesMap[$siteId])) {
                        $projectIdFromProjectSites = $projectSitesMap[$siteId][0];
                        if (isset($projects[$projectIdFromProjectSites])) {
                            $project = $projects[$projectIdFromProjectSites];
                            $projectId = $projectIdFromProjectSites;
                        } else {
                            $directProject = Project::where('id', $projectIdFromProjectSites)
                                ->where('customer_id', $ids['customer_id'])
                                ->where('workspace_id', $ids['workspace_id'])
                                ->select('id', 'title')
                                ->first();
                            if ($directProject) {
                                $project = $directProject;
                                $projectId = $projectIdFromProjectSites;
                                $projects[$projectIdFromProjectSites] = $directProject;
                            }
                        }
                    }
                }
                
                // Second, check site's project_id (fallback if project_sites doesn't have it)
                if (!$project && $attendance->sites && $attendance->sites->project_id) {
                    $siteProjectId = $attendance->sites->project_id;
                    if ($attendance->sites->project && $attendance->sites->project->id) {
                        $project = $attendance->sites->project;
                        $projectId = $attendance->sites->project->id;
                    } elseif (isset($projects[$siteProjectId])) {
                        $project = $projects[$siteProjectId];
                        $projectId = $siteProjectId;
                    } elseif ($siteProjectId) {
                        $directProject = Project::where('id', $siteProjectId)
                            ->where('customer_id', $ids['customer_id'])
                            ->where('workspace_id', $ids['workspace_id'])
                            ->select('id', 'title')
                            ->first();
                        if ($directProject) {
                            $project = $directProject;
                            $projectId = $siteProjectId;
                            $projects[$siteProjectId] = $directProject;
                        }
                    }
                }
                
                // Third, try to get project from roster_assign (lowest priority)
                if (!$project && $rosterAssign && $rosterAssign->project_id) {
                    if (isset($projects[$rosterAssign->project_id])) {
                        $project = $projects[$rosterAssign->project_id];
                        $projectId = $rosterAssign->project_id;
                    } else {
                        $directProject = Project::where('id', $rosterAssign->project_id)
                            ->where('customer_id', $ids['customer_id'])
                            ->where('workspace_id', $ids['workspace_id'])
                            ->select('id', 'title')
                            ->first();
                        if ($directProject) {
                            $project = $directProject;
                            $projectId = $rosterAssign->project_id;
                            $projects[$rosterAssign->project_id] = $directProject;
                        }
                    }
                }
                
                // Filter by project_id if provided
                if ($request->filled('project_id')) {
                    if ($projectId != $request->project_id) {
                        continue; // Skip this attendance if project doesn't match
                    }
                }
                
                // Convert working_hours from minutes to hours (round to 2 decimal places)
                $workingHours = $attendance->working_hours ?? 0;
                if ($workingHours > 0) {
                    $hours = $workingHours / 60;
                    $workingHoursInHours = (float) number_format($hours, 2, '.', '');
                } else {
                    $workingHoursInHours = 0.0;
                }
                
                // Format date according to user's preferred date format
                $formattedDate = Carbon::parse($attendanceDate)->format($userDateFormat);
                
                // Format breaks data
                $breaksData = [];
                if ($attendance->breaks && $attendance->breaks->count() > 0) {
                    foreach ($attendance->breaks as $break) {
                        $breaksData[] = [
                            'id' => $break->id,
                            'break_in' => $break->break_in,
                            'break_out' => $break->break_out,
                            'date' => $break->date ? (is_string($break->date) ? $break->date : Carbon::parse($break->date)->format($userDateFormat)) : null,
                        ];
                    }
                }
                
                $attendanceData[] = [
                    'id' => $attendance->id,
                    'date' => $formattedDate,
                    'check_in' => $attendance->check_in,
                    'check_out' => $attendance->check_out,
                    'working_hours' => $workingHoursInHours,
                    'status' => $attendance->status,
                    'breaks' => $breaksData,
                    'site' => $attendance->sites ? [
                        'id' => $attendance->sites->id,
                        'title' => $attendance->sites->title,
                        'project_id' => $attendance->sites->project_id ?? null,
                    ] : null,
                    'project' => $project ? [
                        'id' => $project->id,
                        'title' => $project->title,
                    ] : null,
                ];
            }
            
            // Determine if filters are applied
            $hasDateFilters = $request->filled('from_date') || $request->filled('to_date');
            $hasSiteFilter = $request->filled('site_id');
            $hasProjectFilter = $request->filled('project_id');
            
            // Only include employee if:
            // 1. They have attendance records matching all filters, OR
            // 2. No filters are applied (show all employees even without attendance)
            if (!empty($attendanceData) || (!$hasDateFilters && !$hasSiteFilter && !$hasProjectFilter)) {
                // Get user_type and subcontractor_id from employee object
                $userType = isset($employee->user_type) ? $employee->user_type : 0; // Default to 0 (internal) if not set
                $subcontractorId = isset($employee->subcontractor_id) ? $employee->subcontractor_id : null;
                $subcontractors = isset($employee->subcontractors) ? $employee->subcontractors : []; // Get subcontractors array
                
                $timesheets[] = [
                    'employee_id' => $employee->id,
                    'employee_xero_id' => $employee->xero_emp_id ?? null,
                    'employee_email' => $employee->employee_email ?? null,
                    'employee_name' => $employee->empPersonalDetails 
                        ? trim(($employee->empPersonalDetails->first_name ?? '') . ' ' . 
                               ($employee->empPersonalDetails->middle_name ?? '') . ' ' . 
                               ($employee->empPersonalDetails->last_name ?? ''))
                        : 'Unknown',
                    'employee_image' => $employee->empPersonalDetails->image ?? null,
                    'attendance_count' => count($attendanceData),
                    'total_working_hours' => (float) number_format(array_sum(array_column($attendanceData, 'working_hours')), 2, '.', ''),
                    'attendances' => $attendanceData,
                    'user_type' => $userType, // 0 = internal, 1 = external/subcontractor
                    'subcontractor_id' => $subcontractorId, // null for internal employees
                    'subcontractors' => $subcontractors, // Array of subcontractors for subcontractor employees
                ];
            }
        }

        return $timesheets;
    }
}
