<?php

namespace App\Jobs;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use App\Http\Controllers\Traits\EmailTrait;
use Illuminate\Support\Facades\Log;

/**
 * Global email notification job
 * 
 * This job can be used to send any email notification in the system.
 * It supports both pre-rendered email content and template rendering.
 * 
 * Usage examples:
 * 
 * 1. Send with pre-rendered content:
 *    SendNotificationEmailJob::dispatch([
 *        'to' => 'user@example.com',
 *        'subject' => 'Subject',
 *        'msg' => view('Emails.template', $data)->render(),
 *        'customer_id' => 1,
 *        'workspace_id' => 1,
 *    ]);
 * 
 * 2. Send with template name and data:
 *    SendNotificationEmailJob::dispatch([
 *        'to' => 'user@example.com',
 *        'subject' => 'Subject',
 *        'template' => 'Emails.template-name',
 *        'template_data' => ['key' => 'value'],
 *        'customer_id' => 1,
 *        'workspace_id' => 1,
 *    ]);
 */
class SendNotificationEmailJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels, EmailTrait;

    /**
     * The number of times the job may be attempted.
     *
     * @var int
     */
    public $tries = 3;

    /**
     * The maximum number of seconds the job can run before timing out.
     *
     * @var int
     */
    public $timeout = 120;

    /**
     * Email parameters
     * 
     * @var array
     */
    protected $emailParams;

    /**
     * Create a new job instance.
     *
     * @param array $emailParams Email parameters containing:
     *   - 'to' (required): Recipient email address
     *   - 'subject' (required): Email subject
     *   - 'msg' (optional): Pre-rendered email HTML content
     *   - 'template' (optional): Blade template path (e.g., 'Emails.template-name')
     *   - 'template_data' (optional): Data array to pass to template (required if 'template' is provided)
     *   - 'customer_id' (optional): Customer ID for email branding
     *   - 'workspace_id' (optional): Workspace ID for email branding
     *   - 'attachments' (optional): Array of attachment data
     */
    public function __construct(array $emailParams)
    {
        $this->emailParams = $emailParams;
    }

    /**
     * Execute the job.
     *
     * @return void
     */
    public function handle()
    {
        Log::info('SendNotificationEmailJob: Job execution started', [
            'to' => $this->emailParams['to'] ?? 'not provided',
            'subject' => $this->emailParams['subject'] ?? 'not provided',
            'has_template' => isset($this->emailParams['template']),
            'has_msg' => isset($this->emailParams['msg']),
            'attempts' => $this->attempts() ?? 1
        ]);

        try {
            // Validate required parameters
            if (empty($this->emailParams['to'])) {
                Log::error('SendNotificationEmailJob: Missing recipient email address');
                return;
            }

            if (empty($this->emailParams['subject'])) {
                Log::error('SendNotificationEmailJob: Missing email subject');
                return;
            }

            // Validate email address
            if (!filter_var($this->emailParams['to'], FILTER_VALIDATE_EMAIL)) {
                Log::warning('SendNotificationEmailJob: Invalid recipient email address', [
                    'to' => $this->emailParams['to']
                ]);
                return;
            }

            // Prepare email content
            $emailContent = null;

            // If template is provided, render it
            if (isset($this->emailParams['template'])) {
                if (empty($this->emailParams['template_data'])) {
                    Log::error('SendNotificationEmailJob: Template provided but template_data is missing', [
                        'template' => $this->emailParams['template']
                    ]);
                    return;
                }

                try {
                    $emailContent = view($this->emailParams['template'], $this->emailParams['template_data'])->render();
                    Log::info('SendNotificationEmailJob: Template rendered successfully', [
                        'template' => $this->emailParams['template']
                    ]);
                } catch (\Exception $e) {
                    Log::error('SendNotificationEmailJob: Failed to render template', [
                        'template' => $this->emailParams['template'],
                        'error' => $e->getMessage(),
                        'trace' => $e->getTraceAsString()
                    ]);
                    throw $e;
                }
            } 
            // If pre-rendered message is provided, use it
            elseif (isset($this->emailParams['msg'])) {
                $emailContent = $this->emailParams['msg'];
                Log::info('SendNotificationEmailJob: Using pre-rendered email content');
            } 
            // Neither template nor msg provided
            else {
                Log::error('SendNotificationEmailJob: Neither template nor msg provided');
                return;
            }

            // Prepare email parameters for SendInstantEmail
            $params = [
                'to' => $this->emailParams['to'],
                'subject' => $this->emailParams['subject'],
                'msg' => $emailContent,
            ];

            // Add customer_id and workspace_id if provided
            if (isset($this->emailParams['customer_id'])) {
                $params['customer_id'] = $this->emailParams['customer_id'];
            }

            if (isset($this->emailParams['workspace_id'])) {
                $params['workspace_id'] = $this->emailParams['workspace_id'];
            }

            // Add attachments if provided
            if (isset($this->emailParams['attachments']) && is_array($this->emailParams['attachments'])) {
                $params['attachments'] = $this->emailParams['attachments'];
            }

            Log::info('SendNotificationEmailJob: Calling SendInstantEmail', [
                'to' => $params['to'],
                'subject' => $params['subject'],
                'has_customer_id' => isset($params['customer_id']),
                'has_workspace_id' => isset($params['workspace_id']),
                'has_attachments' => isset($params['attachments'])
            ]);

            // Send the email
            $emailResult = $this->SendInstantEmail($params);

            Log::info('SendNotificationEmailJob: SendInstantEmail completed', [
                'to' => $params['to'],
                'result' => $emailResult ?? 'no_return_value'
            ]);

            return $emailResult;
        } catch (\Exception $e) {
            Log::error('SendNotificationEmailJob: Failed to send email', [
                'to' => $this->emailParams['to'] ?? 'unknown',
                'subject' => $this->emailParams['subject'] ?? 'unknown',
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
            throw $e; // Re-throw to mark job as failed and trigger retry
        }
    }

    /**
     * Handle a job failure.
     *
     * @param \Throwable $exception
     * @return void
     */
    public function failed(\Throwable $exception)
    {
        Log::error('SendNotificationEmailJob: Job failed after all retries', [
            'to' => $this->emailParams['to'] ?? 'unknown',
            'subject' => $this->emailParams['subject'] ?? 'unknown',
            'error' => $exception->getMessage(),
            'attempts' => $this->attempts() ?? 0
        ]);
    }
}
