<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Laravel\Passport\HasApiTokens;
use App\Models\SubcontractorCompany;

class EmployeeSubcontractor extends Authenticatable
{
    use HasFactory, Notifiable, HasApiTokens;

    protected $table = 'employees_subcontractors';

    protected $guard = 'subcontractor_employee';

    protected $fillable = [
        'first_name',
        'middle_name',
        'last_name',
        'email',
        'password',
        'force_reset',
        'phone',
        'address',
        'state',
        'suburb',
        'dob',
        'work_experience',
        'trade_qualified',
        'trade_qualified_year',
        'trade_licensed',
        'trade_licensed_year',
        'year_commenced',
        'citizenship_status',
        'visa_attachement',
        'allergies',
        'previous_injuries',
        'medical_condition',
        'medical_attachement_files',
        'signature',
        'profile_image',
        'relation',
        'emergency_name',
        'mobile',
        'xero_emp_id',
    ];

    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * The accessors to append to the model's array form.
     *
     * @var array
     */
    protected $appends = [
        'subcontractors',
    ];

    /**
     * Find user for Passport authentication by email
     */
    public function findForPassport($username)
    {
        return $this->where('email', $username)->first();
    }

    /**
     * Get the associations for this employee.
     */
    public function associations()
    {
        return $this->hasMany(EmployeeSubcontractorMeta::class, 'emp_id');
    }

    /**
     * Accessor: Get array of subcontractors with id and name.
     * This will automatically be included when the model is serialized to JSON.
     * Returns subcontractors associated with this employee through the associations table.
     * Only returns subcontractors that have required_docs_status = 1 AND induction_status = 1.
     * 
     * Filters by authenticated customer's customer_id and workspace_id.
     * Checks subcontractor's status from subcontractor_companies table (required_docs_status and induction_status columns).
     *
     * @return array Array of subcontractor information with id, name, company_name, and active status
     */
    public function getSubcontractorsAttribute()
    {
        // Get authenticated customer context
        $customerId = null;
        $workspaceId = null;
        
        if (Auth::check()) {
            $user = Auth::user();
            
            // Check if authenticated user is a customer
            if ($user instanceof \App\Models\User && $user->user_type != config('constants.user_types.subcontractor')) {
                $customerId = $user->id;
                $workspaceId = $user->current_workspace_id;
            }
            // Check if authenticated user is an employee (has customer_id)
            elseif ($user instanceof \App\Models\EmpCompanyDetails) {
                $customerId = $user->customer_id;
                $workspaceId = $user->workspace_id;
            }
        }
        
        // If no customer context, return empty array
        if (!$customerId || !$workspaceId) {
            return [];
        }
        
        return $this->associations()
            ->with('subcontractor:id,name,company_name')
            ->get()
            ->map(function ($association) use ($customerId, $workspaceId) {
                $subcontractor = $association->subcontractor;
                if (!$subcontractor) {
                    return null;
                }
                
                // Get subcontractor company association for this customer/workspace
                $subcontractorCompany = SubcontractorCompany::where('user_id', $subcontractor->id)
                    ->where('customer_id', $customerId)
                    ->where('workspace_id', $workspaceId)
                    ->where('del', '0')
                    ->first();
                
                // If subcontractor is not associated with this customer/workspace, skip
                if (!$subcontractorCompany) {
                    return null;
                }
                
                // Check required_docs_status = 1 (complete)
                if ($subcontractorCompany->required_docs_status != SubcontractorCompany::STATUS_COMPLETE) {
                    return null;
                }
                
                // Check induction_status = 1 (complete) - uses the column directly
                if ($subcontractorCompany->induction_status != SubcontractorCompany::INDUCTION_COMPLETE) {
                    return null;
                }
                
                return [
                    'id' => $subcontractor->id,
                    'name' => $subcontractor->name ?? $subcontractor->company_name ?? null,
                    'active' => $association->active,
                ];
            })
            ->filter() // Remove null values
            ->values() // Re-index array
            ->toArray();
    }
}


