<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class Tender extends BaseModel
{
    use HasFactory;

    protected $fillable = [
        'title',
        'reference',
        'project_id',
        'closing_date',
        'status',
        'tender_issued_date',
        'rfi_cut_off_date',
        'scope_of_work',
        'reminder_days_before',
        'allow_late_submissions',
        'allow_bidder_document_uploads',
        'submitted_at',
        'bidder_id',
        'total_base_bid',
        'exclusions_clarifications',
        'is_late_submission',
        'customer_id',
        'workspace_id',
        'is_published',
    ];

    protected $hidden = [
        'customer_id',
        'workspace_id',
    ];

    /**
     * Boot method to auto-generate unique reference
     */
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($tender) {
            if (empty($tender->reference)) {
                $tender->reference = static::generateUniqueReference($tender);
            }
        });
    }

  
    protected static function extractCompanyInitials($companyName)
    {
        if (empty($companyName)) {
            return 'XY'; // Default fallback
        }

        // Remove special characters and split by spaces
        $words = preg_split('/[\s\-_]+/', trim($companyName));
        $initials = '';

        foreach ($words as $word) {
            // Get first letter of each word (uppercase)
            $firstChar = strtoupper(substr(trim($word), 0, 1));
            if (!empty($firstChar) && ctype_alpha($firstChar)) {
                $initials .= $firstChar;
                // Limit to 2 characters for company initials
                if (strlen($initials) >= 2) {
                    break;
                }
            }
        }

        // If we don't have 2 characters, pad or use first 2 characters
        if (strlen($initials) < 2) {
            $cleanName = preg_replace('/[^A-Za-z]/', '', $companyName);
            $initials = strtoupper(substr($cleanName, 0, 2));
            if (strlen($initials) < 2) {
                $initials = str_pad($initials, 2, 'X', STR_PAD_RIGHT);
            }
        }

        return $initials;
    }


    protected static function extractTitleInitials($title)
    {
        if (empty($title)) {
            return 'XYZ'; // Default fallback
        }

        // Split by spaces and get first letter of each word
        $words = preg_split('/[\s\-_]+/', trim($title));
        $initials = '';

        foreach ($words as $word) {
            // Get first letter of each word (uppercase)
            $firstChar = strtoupper(substr(trim($word), 0, 1));
            if (!empty($firstChar) && ctype_alpha($firstChar)) {
                $initials .= $firstChar;
                // Limit to 3 characters
                if (strlen($initials) >= 3) {
                    break;
                }
            }
        }

        // If we don't have 3 characters, pad with X or use first 3 letters as fallback
        if (strlen($initials) < 3) {
            $cleanTitle = preg_replace('/[^A-Za-z0-9]/', '', $title);
            $fallback = strtoupper(substr($cleanTitle, 0, 3));
            if (strlen($fallback) >= strlen($initials)) {
                $initials = $fallback;
            }
            // If still less than 3, pad with X
            if (strlen($initials) < 3) {
                $initials = str_pad($initials, 3, 'X', STR_PAD_RIGHT);
            }
        }

        return $initials;
    }

    /**
     * Generate unique reference in format: CW-PT-STT-MM-YYYY-XXXXX
     * Where:
     * - CW: Customer company name's first letters (2 chars)
     * - PT: Static text
     * - STT: Tender title starting words (3 chars)
     * - MM-YYYY: Month and year
     * - XXXXX: 5 digit unique sequential number
     */
    protected static function generateUniqueReference($tender)
    {
        // Get customer company name from adminsettings
        $customerInitials = 'XX'; // Default fallback
        if (!empty($tender->customer_id)) {
            $companyName = null;
            
            // Try to get from adminsettings first
            if (!empty($tender->workspace_id)) {
                $companyName = DB::table('adminsettings')
                    ->where('customer_id', $tender->customer_id)
                    ->where('workspace', $tender->workspace_id)
                    ->where('key', 'company_company_name')
                    ->value('value');
            }
            
            // If not found with workspace, try without workspace filter
            if (empty($companyName)) {
                $companyName = DB::table('adminsettings')
                    ->where('customer_id', $tender->customer_id)
                    ->where('key', 'company_company_name')
                    ->value('value');
            }
            
            // If found in adminsettings, use it
            if (!empty($companyName)) {
                $customerInitials = static::extractCompanyInitials($companyName);
            } else {
                // Fallback to User model company_name or name
                $customer = User::find($tender->customer_id);
                if ($customer && !empty($customer->company_name)) {
                    $customerInitials = static::extractCompanyInitials($customer->company_name);
                } elseif ($customer && !empty($customer->name)) {
                    $customerInitials = static::extractCompanyInitials($customer->name);
                }
            }
        }

        // Get tender title initials (first 3 letters)
        $titleInitials = static::extractTitleInitials($tender->title ?? '');

        // Get month and year
        $month = Carbon::now()->format('m');
        $year = Carbon::now()->format('Y');

        // Build prefix: CW-PT-STT-MM-YYYY
        $prefix = "{$customerInitials}-PT-{$titleInitials}-{$month}-{$year}";
        
        // Generate random 5-character alphanumeric string (numbers and letters)
        $characters = '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ';
        $maxAttempts = 100; // Prevent infinite loop
        $attempts = 0;
        $reference = '';
        
        do {
            $randomString = '';
            // Generate 5 random characters
            for ($i = 0; $i < 5; $i++) {
                $randomString .= $characters[random_int(0, strlen($characters) - 1)];
            }
            $reference = "{$prefix}-{$randomString}";
            $attempts++;
            
            // Safety check to prevent infinite loop
            if ($attempts >= $maxAttempts) {
                // Fallback: use timestamp-based random string with alphanumeric
                $randomString = strtoupper(substr(md5(uniqid(rand(), true) . time()), 0, 5));
                // Ensure it contains both letters and numbers by mixing
                $randomString = str_shuffle($randomString);
                $reference = "{$prefix}-{$randomString}";
                break;
            }
        } while (static::where('reference', $reference)->exists());
        
        return $reference;
    }

    /**
     * Get the project associated with the tender.
     */
    public function project()
    {
        return $this->belongsTo(Project::class, 'project_id');
    }

    /**
     * Get the bidder (user) associated with the tender.
     */
    public function bidder()
    {
        return $this->belongsTo(User::class, 'bidder_id');
    }

    /**
     * Get all line items for this tender.
     */
    public function lineItems()
    {
        return $this->hasMany(TenderLineItem::class, 'tender_id');
    }

    /**
     * Get all bids for this tender.
     */
    public function bids()
    {
        return $this->hasMany(TenderLineItemsBid::class, 'tender_id');
    }

    /**
     * Get all participants for this tender.
     */
    public function participants()
    {
        return $this->hasMany(TenderParticipant::class, 'tender_id');
    }

    /**
     * Get all RFIs for this tender.
     */
    public function rfis()
    {
        return $this->hasMany(TenderRfi::class, 'tender_id');
    }

    /**
     * Get all attachments for this tender (polymorphic).
     */
    public function attachments()
    {
        return $this->morphMany(TenderAttachment::class, 'entity', 'entity_type', 'entity_id');
    }
}

