<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class GoHighLevelService
{
    private $apiKey;
    private $locationId;
    private $baseUrl = 'https://services.gohighlevel.com/v1';
    private $enabled;

    /**
     * Constructor - Initialize GoHighLevel service with credentials from config
     */
    public function __construct()
    {
        $this->apiKey = config('services.gohighlevel.api_key');
        $this->locationId = config('services.gohighlevel.location_id');
        $this->enabled = !empty($this->apiKey) && !empty($this->locationId);

        if (!$this->enabled) {
            Log::warning('GoHighLevel service is not configured. Missing API key or Location ID.');
        }
    }

    /**
     * Check if GoHighLevel is enabled and configured
     *
     * @return bool
     */
    public function isEnabled(): bool
    {
        return $this->enabled;
    }

    /**
     * Create or update a contact in GoHighLevel
     *
     * @param array $contactData Contact data including:
     *   - firstName (required)
     *   - lastName (required)
     *   - email (required)
     *   - phone (required)
     *   - companyName (optional)
     *   - customField (optional) - array of custom field key-value pairs
     *   - tags (optional) - array of tag strings
     *   - source (optional) - source of the contact
     *
     * @return array|null Returns contact data on success, null on failure
     * @throws \Exception
     */
    public function createOrUpdateContact(array $contactData): ?array
    {
        if (!$this->isEnabled()) {
            Log::warning('GoHighLevel: Attempted to create contact but service is not enabled');
            return null;
        }

        try {
            // Prepare the contact payload according to GoHighLevel API
            $payload = [
                'firstName' => $contactData['firstName'] ?? '',
                'lastName' => $contactData['lastName'] ?? '',
                'email' => $contactData['email'] ?? '',
                'phone' => $contactData['phone'] ?? '',
            ];

            // Add company name if provided
            if (!empty($contactData['companyName'])) {
                $payload['companyName'] = $contactData['companyName'];
            }

            // Add source if provided
            if (!empty($contactData['source'])) {
                $payload['source'] = $contactData['source'];
            }

            // Add custom fields if provided
            if (!empty($contactData['customField']) && is_array($contactData['customField'])) {
                $payload['customField'] = $contactData['customField'];
            }

            // Add tags if provided
            if (!empty($contactData['tags']) && is_array($contactData['tags'])) {
                $payload['tags'] = $contactData['tags'];
            }

            // Make API request to create/update contact
            // Location ID is passed as query parameter in GoHighLevel API
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey,
                'Content-Type' => 'application/json',
                'Version' => '2021-07-28',
            ])->post("{$this->baseUrl}/contacts/?locationId={$this->locationId}", $payload);

            $statusCode = $response->status();
            $responseData = $response->json();

            // Log the request for debugging
            Log::info('GoHighLevel: Contact creation request', [
                'location_id' => $this->locationId,
                'email' => $contactData['email'] ?? null,
                'status_code' => $statusCode,
                'response' => $responseData,
            ]);

            // Check if request was successful
            if ($response->successful()) {
                Log::info('GoHighLevel: Contact created/updated successfully', [
                    'contact_id' => $responseData['contact']['id'] ?? null,
                    'email' => $contactData['email'] ?? null,
                ]);
                return $responseData['contact'] ?? $responseData;
            }

            // Handle API errors
            $errorMessage = $this->extractErrorMessage($responseData, $statusCode);
            Log::error('GoHighLevel: Failed to create/update contact', [
                'status_code' => $statusCode,
                'error' => $errorMessage,
                'response' => $responseData,
                'payload' => $payload,
            ]);

            // Don't throw exception - let the calling code decide how to handle
            return null;

        } catch (\Illuminate\Http\Client\ConnectionException $e) {
            Log::error('GoHighLevel: Connection error', [
                'error' => $e->getMessage(),
                'email' => $contactData['email'] ?? null,
            ]);
            return null;
        } catch (\Exception $e) {
            Log::error('GoHighLevel: Unexpected error creating contact', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'email' => $contactData['email'] ?? null,
            ]);
            return null;
        }
    }

    /**
     * Create an opportunity (add contact to pipeline)
     *
     * @param string $contactId GoHighLevel contact ID
     * @param string|null $pipelineId Pipeline ID (optional, from config if not provided)
     * @param string|null $stageId Stage ID (optional, from config if not provided)
     * @param string|null $title Opportunity title
     *
     * @return array|null Returns opportunity data on success, null on failure
     */
    public function createOpportunity(
        string $contactId,
        ?string $pipelineId = null,
        ?string $stageId = null,
        ?string $title = null
    ): ?array {
        if (!$this->isEnabled()) {
            return null;
        }

        // Use pipeline/stage from config if not provided
        $pipelineId = $pipelineId ?? config('services.gohighlevel.pipeline_id');
        $stageId = $stageId ?? config('services.gohighlevel.stage_id');

        // If no pipeline configured, skip opportunity creation
        if (empty($pipelineId)) {
            Log::info('GoHighLevel: Skipping opportunity creation - no pipeline configured');
            return null;
        }

        try {
            $payload = [
                'contactId' => $contactId,
                'pipelineId' => $pipelineId,
            ];

            if (!empty($stageId)) {
                $payload['stageId'] = $stageId;
            }

            if (!empty($title)) {
                $payload['title'] = $title;
            }

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey,
                'Content-Type' => 'application/json',
                'Version' => '2021-07-28',
            ])->post("{$this->baseUrl}/opportunities/?locationId={$this->locationId}", $payload);

            $statusCode = $response->status();
            $responseData = $response->json();

            if ($response->successful()) {
                Log::info('GoHighLevel: Opportunity created successfully', [
                    'contact_id' => $contactId,
                    'pipeline_id' => $pipelineId,
                    'opportunity_id' => $responseData['opportunity']['id'] ?? null,
                ]);
                return $responseData['opportunity'] ?? $responseData;
            }

            $errorMessage = $this->extractErrorMessage($responseData, $statusCode);
            Log::warning('GoHighLevel: Failed to create opportunity', [
                'status_code' => $statusCode,
                'error' => $errorMessage,
                'contact_id' => $contactId,
            ]);

            return null;

        } catch (\Exception $e) {
            Log::error('GoHighLevel: Error creating opportunity', [
                'error' => $e->getMessage(),
                'contact_id' => $contactId,
            ]);
            return null;
        }
    }

    /**
     * Extract error message from GoHighLevel API response
     *
     * @param array|null $responseData
     * @param int $statusCode
     * @return string
     */
    private function extractErrorMessage(?array $responseData, int $statusCode): string
    {
        if (empty($responseData)) {
            return "HTTP {$statusCode} - No response data";
        }

        // GoHighLevel error format
        if (isset($responseData['message'])) {
            return $responseData['message'];
        }

        if (isset($responseData['error'])) {
            if (is_string($responseData['error'])) {
                return $responseData['error'];
            }
            if (is_array($responseData['error']) && isset($responseData['error']['message'])) {
                return $responseData['error']['message'];
            }
        }

        // Check for validation errors
        if (isset($responseData['errors']) && is_array($responseData['errors'])) {
            $errors = [];
            foreach ($responseData['errors'] as $field => $messages) {
                if (is_array($messages)) {
                    $errors[] = "{$field}: " . implode(', ', $messages);
                } else {
                    $errors[] = "{$field}: {$messages}";
                }
            }
            if (!empty($errors)) {
                return implode(' | ', $errors);
            }
        }

        return "HTTP {$statusCode} - Unknown error";
    }

    /**
     * Sync demo request to GoHighLevel
     * This is a convenience method that combines contact creation and opportunity creation
     *
     * @param array $demoRequestData Demo request data
     * @return array|null Returns array with 'contact' and optionally 'opportunity' keys, or null on failure
     */
    public function syncDemoRequest(array $demoRequestData): ?array
    {
        if (!$this->isEnabled()) {
            return null;
        }

        try {
            // Map demo request fields to GoHighLevel contact fields
            $contactData = [
                'firstName' => $demoRequestData['first_name'] ?? '',
                'lastName' => $demoRequestData['last_name'] ?? '',
                'email' => $demoRequestData['work_email'] ?? '',
                'phone' => $demoRequestData['phone'] ?? '',
                'companyName' => $demoRequestData['company_name'] ?? '',
                'source' => 'Demo Request Form',
            ];

            // Prepare custom fields
            $customFields = [];
            
            // Contact Type (maps to company_type from demo request)
            if (!empty($demoRequestData['company_type'])) {
                $customFields['type'] = $demoRequestData['company_type'];
            }

            // Company Size
            if (!empty($demoRequestData['company_size'])) {
                $customFields['company_size'] = (string) $demoRequestData['company_size'];
            }

            // Email Consent
            if (isset($demoRequestData['email_consent'])) {
                $customFields['email_consent'] = $demoRequestData['email_consent'] ? 'Yes' : 'No';
            }

            if (!empty($customFields)) {
                $contactData['customField'] = $customFields;
            }

            // Add tags
            $tags = ['Demo Request'];
            if (!empty($demoRequestData['company_type'])) {
                $tags[] = 'Company Type: ' . $demoRequestData['company_type'];
            }
            $contactData['tags'] = $tags;

            // Create or update contact
            $contact = $this->createOrUpdateContact($contactData);

            if (!$contact) {
                return null;
            }

            $result = ['contact' => $contact];

            // Create opportunity if contact was created successfully
            $contactId = $contact['id'] ?? null;
            if ($contactId) {
                $opportunity = $this->createOpportunity(
                    $contactId,
                    null, // Use pipeline from config
                    null, // Use stage from config
                    'Demo Request - ' . ($demoRequestData['company_name'] ?? 'New Lead')
                );

                if ($opportunity) {
                    $result['opportunity'] = $opportunity;
                }
            }

            return $result;

        } catch (\Exception $e) {
            Log::error('GoHighLevel: Error syncing demo request', [
                'error' => $e->getMessage(),
                'demo_request_data' => $demoRequestData,
            ]);
            return null;
        }
    }
}