<?php

namespace App\Services;

use GuzzleHttp\Client;
use Illuminate\Support\Facades\Log;
use App\Models\XeroToken;
use App\Models\Adminsettings;
use App\Models\EmployeeAttendance;
use Carbon\Carbon;
use Illuminate\Support\Facades\Cache;

class XeroService
{
    private $client;
    private $clientId;
    private $clientSecret;
    private $redirectUri;
    private $baseUrl = 'https://api.xero.com';
    private $authUrl = 'https://login.xero.com/identity/connect/authorize';
    private $tokenUrl = 'https://identity.xero.com/connect/token';
    private $customerId;
    private $workspaceId;

    public function __construct($customerId = null, $workspaceId = null)
    {
        $this->client = new Client();
        $this->customerId = $customerId;
        $this->workspaceId = $workspaceId;
        $this->loadCredentials($customerId, $workspaceId);
        $this->redirectUri = config('services.xero.redirect_uri', config('app.url') . '/api/redirect');
        Log::info('XeroService constructed', [
            'customer_id' => $customerId,
            'workspace_id' => $workspaceId,
            'redirect_uri' => $this->redirectUri,
        ]);
    }

    /**
     * Load Xero credentials from adminsettings table
     */
    private function loadCredentials($customerId, $workspaceId)
    {
        // Get client_id from adminsettings
        $clientIdSetting = Adminsettings::where('customer_id', $customerId)
            ->where('workspace', $workspaceId)
            ->where('key', 'company_xero_client_id')
            ->first();
        $secretIdSetting = Adminsettings::where('customer_id', $customerId)
            ->where('workspace', $workspaceId)
            ->where('key', 'company_xero_secret_id')
            ->first();
        if ($clientIdSetting && $secretIdSetting) {
            $this->clientId = $clientIdSetting->value;
            $this->clientSecret = $secretIdSetting->value;
        }
        else {
            Log::error('Xero credentials not found for customer_id: ' . $customerId . ' and workspace_id: ' . $workspaceId);
            return false;
        }
    }


    /**
     * Generate authorization URL
     */
    public function getAuthorizationUrl($state = null)
    {
        $state = $state ?? bin2hex(random_bytes(16));

        // IMPORTANT:
        // - redirect_uri must match EXACTLY the OAuth 2.0 redirect URI configured in Xero (same as Postman guide)
        // - scopes should come from config/env so they match the Xero guide & can be changed without code
        $redirectUri = $this->redirectUri;
        $scopes = config('services.xero.scopes');
        $params = [
            'response_type' => 'code',
            'client_id' => $this->clientId,
            'redirect_uri' => $redirectUri,
            'scope' => $scopes,
            'state' => $state,
        ];
        $url = $this->authUrl . '?' . http_build_query($params);
        Log::info('Xero authorization URL generated', [
            'redirect_uri' => $redirectUri,
            'state' => $state,
            'scopes' => $scopes,
        ]);
        return $url;
    }

    /**
     * Exchange authorization code for tokens
     * @param string $code Authorization code from Xero
     * @param int|null $customerId Customer ID to store in token record (from state parameter)
     * @param int|null $workspaceId Workspace ID to store in token record (from state parameter)
     */
    public function exchangeCodeForTokens($code, $customerId = null, $workspaceId = null)
    {
        try {
            $response = $this->client->post($this->tokenUrl, [
                'headers' => [
                    'Content-Type' => 'application/x-www-form-urlencoded',
                ],
                'form_params' => [
                    'grant_type' => 'authorization_code',
                    'client_id' => $this->clientId,
                    'client_secret' => $this->clientSecret,
                    'code' => $code,
                    'redirect_uri' => $this->redirectUri,
                ],
            ]);
            $responseBody = $response->getBody()->getContents();
            $tokenData = json_decode($responseBody, true);
            if (isset($tokenData['error'])) {
                $errorMessage = $tokenData['error_description'] ?? $tokenData['error'];
                Log::error('Xero token exchange error response', [
                    'error' => $tokenData['error'],
                    'error_description' => $errorMessage,
                    'response' => $tokenData,
                ]);
                throw new \Exception('Xero API Error: ' . $errorMessage);
            }
            if (isset($tokenData['access_token'])) {
                // Calculate expiration time
                $expiresAt = now()->addSeconds($tokenData['expires_in'] ?? 1800);
                // Save token with customer_id and workspace_id (stored for our records, not sent to Xero)
                // Deactivate all other tokens and create new active token
                XeroToken::where('is_active', true)->update(['is_active' => false]);
                $xeroToken = XeroToken::create([
                    'customer_id' => $customerId,
                    'workspace_id' => $workspaceId,
                    'access_token' => $tokenData['access_token'],
                    'refresh_token' => $tokenData['refresh_token'] ?? null,
                    'id_token' => $tokenData['id_token'] ?? null,
                    'expires_in' => $tokenData['expires_in'] ?? 1800,
                    'expires_at' => $expiresAt,
                    'token_type' => $tokenData['token_type'] ?? 'Bearer',
                    'is_active' => true,
                ]);
                // Get tenant information
                $this->fetchAndStoreTenants($xeroToken, $tokenData['access_token']);
                Log::info('Xero token exchange successful', [
                    'token_id' => $xeroToken->id,
                    'tenant_id' => $xeroToken->tenant_id,
                ]);
                return $xeroToken;
            }
            Log::error('Xero token exchange failed - no access_token in response', [
                'response' => $tokenData,
            ]);
            throw new \Exception('Failed to exchange code for tokens. No access_token received.');
        } catch (\GuzzleHttp\Exception\ClientException $e) {
            $response = $e->getResponse();
            $errorBody = $response ? $response->getBody()->getContents() : 'No response body';
            $errorData = json_decode($errorBody, true);
            Log::error('Xero token exchange HTTP error', [
                'status_code' => $response ? $response->getStatusCode() : null,
                'error_body' => $errorBody,
                'error_data' => $errorData,
            ]);
            $errorMessage = isset($errorData['error_description'])
                ? $errorData['error_description']
                : ($errorData['error'] ?? $e->getMessage());

            throw new \Exception('Xero API Error: ' . $errorMessage);
        } catch (\Exception $e) {
            Log::error('Xero token exchange error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
            ]);
            throw $e;
        }
    }

    /**
     * Refresh access token
     */
    public function refreshToken($xeroToken)
    {
        try {
            if (!$xeroToken->refresh_token) {
                throw new \Exception('No refresh token available. Please re-authenticate.');
            }
            // CRITICAL: Load credentials for the customer/workspace that owns this token
            // The token was created with specific credentials, so we must use the same ones for refresh
            if ($xeroToken->customer_id && $xeroToken->workspace_id) {
                $this->loadCredentials($xeroToken->customer_id, $xeroToken->workspace_id);
            }
            // Verify credentials are loaded
            if (!$this->clientId || !$this->clientSecret) {
                Log::error('Xero credentials not found for token refresh', [
                    'token_id' => $xeroToken->id,
                    'customer_id' => $xeroToken->customer_id,
                    'workspace_id' => $xeroToken->workspace_id,
                ]);
                throw new \Exception('Xero credentials not configured for this customer/workspace. Please check settings.');
            }
            Log::info('Refreshing Xero token', [
                'token_id' => $xeroToken->id,
                'customer_id' => $xeroToken->customer_id,
                'workspace_id' => $xeroToken->workspace_id,
                'expires_at' => $xeroToken->expires_at,
                'client_id_loaded' => !empty($this->clientId),
            ]);
            $response = $this->client->post($this->tokenUrl, [
                'headers' => [
                    'Content-Type' => 'application/x-www-form-urlencoded',
                ],
                'form_params' => [
                    'grant_type' => 'refresh_token',
                    'client_id' => $this->clientId,
                    'client_secret' => $this->clientSecret,
                    'refresh_token' => $xeroToken->refresh_token,
                    'redirect_uri' => $this->redirectUri,
                ],
            ]);
            $tokenData = json_decode($response->getBody()->getContents(), true);
            if (isset($tokenData['access_token'])) {
                $expiresAt = now()->addSeconds($tokenData['expires_in'] ?? 1800);
                // Update token while preserving customer_id and workspace_id from existing token
                $xeroToken->update([
                    'access_token' => $tokenData['access_token'],
                    'refresh_token' => $tokenData['refresh_token'] ?? $xeroToken->refresh_token,
                    'id_token' => $tokenData['id_token'] ?? $xeroToken->id_token,
                    'expires_in' => $tokenData['expires_in'] ?? 1800,
                    'expires_at' => $expiresAt,
                    'is_active' => true,
                ]);
                // Reload the model to get updated attributes
                $xeroToken->refresh();
                Log::info('Xero token refreshed successfully', [
                    'token_id' => $xeroToken->id,
                    'new_expires_at' => $xeroToken->expires_at,
                ]);
                return $xeroToken;
            }
            Log::error('Xero token refresh failed - no access_token in response', [
                'response' => $tokenData,
            ]);
            throw new \Exception('Failed to refresh token: No access_token in response');
        } catch (\GuzzleHttp\Exception\ClientException $e) {
            $response = $e->getResponse();
            $errorBody = $response ? $response->getBody()->getContents() : 'No response body';
            $errorData = json_decode($errorBody, true);
            Log::error('Xero token refresh HTTP error', [
                'status_code' => $response ? $response->getStatusCode() : null,
                'error_body' => $errorBody,
                'error_data' => $errorData,
            ]);
            $errorMessage = isset($errorData['error_description'])
                ? $errorData['error_description']
                : ($errorData['error'] ?? $e->getMessage());
            throw new \Exception('Xero token refresh failed: ' . $errorMessage);
        } catch (\Exception $e) {
            Log::error('Xero token refresh error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
            ]);
            throw $e;
        }
    }

    /**
     * Get connected organizations (tenants)
     */
    public function getConnections($accessToken)
    {
        try {
            $response = $this->client->get($this->baseUrl . '/connections', [
                'headers' => [
                    'Authorization' => 'Bearer ' . $accessToken,
                    'Accept' => 'application/json',
                ],
            ]);
            return json_decode($response->getBody()->getContents(), true);
        } catch (\Exception $e) {
            Log::error('Xero get connections error: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Fetch and store tenant information
     */
    private function fetchAndStoreTenants($xeroToken, $accessToken)
    {
        try {
            $connections = $this->getConnections($accessToken);
            if (!empty($connections) && is_array($connections)) {
                // Store the first tenant (you can modify this to handle multiple tenants)
                $tenant = $connections[0] ?? null;
                if ($tenant) {
                    $xeroToken->update([
                        'tenant_id' => $tenant['tenantId'] ?? null,
                        'tenant_name' => $tenant['tenantName'] ?? null,
                        'tenant_type' => $tenant['tenantType'] ?? null,
                    ]);
                }
            }
        } catch (\Exception $e) {
            Log::error('Xero fetch tenants error: ' . $e->getMessage());
            // Don't throw, just log the error
        }
    }

    /**
     * Get valid access token (refresh if needed)
     */
    public function getValidAccessToken($customerId = null, $workspaceId = null)
    {
        // If customer/workspace provided, reload credentials
        if ($customerId && $workspaceId && ($customerId != $this->customerId || $workspaceId != $this->workspaceId)) {
            $this->loadCredentials($customerId, $workspaceId);
        }
        // Get token for this customer/workspace
        $xeroToken = null;
        if ($customerId && $workspaceId) {
            $xeroToken = XeroToken::where('customer_id', $customerId)
                ->where('workspace_id', $workspaceId)
                ->where('is_active', true)
                ->orderBy('created_at', 'desc')
                ->first();
        } else {
            $xeroToken = XeroToken::getActiveToken();
        }
        if (!$xeroToken) {
            throw new \Exception('No Xero token found. Please authenticate first.');
        }
        // Check if token is expired and refresh if needed
        if ($xeroToken->isExpired()) {
            // Reload credentials from token's customer/workspace if different
            if ($xeroToken->customer_id && $xeroToken->workspace_id) {
                $this->loadCredentials($xeroToken->customer_id, $xeroToken->workspace_id);
            }   
            // Refresh the token
            $refreshedToken = $this->refreshToken($xeroToken);
            // Reload from database to ensure we have the latest values
            $xeroToken = $refreshedToken->fresh();
            Log::info('Token refreshed and reloaded', [
                'token_id' => $xeroToken->id,
                'new_expires_at' => $xeroToken->expires_at,
            ]);
        }
        return $xeroToken->access_token;
    }

    /**
     * Make authenticated API request to Xero
     */
    public function makeRequest($method, $endpoint, $options = [], $customerId = null, $workspaceId = null)
    {
        // Use the customer/workspace from the service instance if not provided
        if (!$customerId || !$workspaceId) {
            $customerId = $this->customerId;
            $workspaceId = $this->workspaceId;
        }
        // Get token first to check if it exists
        $xeroToken = null;
        if ($customerId && $workspaceId) {
            $xeroToken = XeroToken::where('customer_id', $customerId)
                ->where('workspace_id', $workspaceId)
                ->where('is_active', true)
                ->orderBy('created_at', 'desc')
                ->first();
        } else {
            $xeroToken = XeroToken::getActiveToken();
        }
        if (!$xeroToken) {
            throw new \Exception('No Xero token found. Please authenticate first.');
        }
        // Get valid access token (this will refresh if expired)
        // After refresh, get the updated token for tenant_id
        $accessToken = $this->getValidAccessToken($customerId, $workspaceId);
        // Get the latest token after potential refresh (for tenant_id)
        if ($customerId && $workspaceId) {
            $xeroToken = XeroToken::where('customer_id', $customerId)
                ->where('workspace_id', $workspaceId)
                ->where('is_active', true)
                ->orderBy('created_at', 'desc')
                ->first();
        } else {
            $xeroToken = XeroToken::getActiveToken();
        }
        
        // Validate tenant_id is present
        if (!$xeroToken->tenant_id) {
            Log::error('Xero tenant_id is missing', [
                'token_id' => $xeroToken->id,
                'customer_id' => $customerId,
                'workspace_id' => $workspaceId,
                'endpoint' => $endpoint,
            ]);
            // Try to fetch tenant information if missing
            try {
                $this->fetchAndStoreTenants($xeroToken, $accessToken);
                // Reload token to get updated tenant_id
                $xeroToken->refresh();
            } catch (\Exception $e) {
                Log::error('Failed to fetch Xero tenant information', [
                    'token_id' => $xeroToken->id,
                    'error' => $e->getMessage(),
                ]);
            }
            
            // If still no tenant_id, throw an error
            if (!$xeroToken->tenant_id) {
                throw new \Exception('Xero tenant_id is missing. Please reconnect your Xero account from settings.');
            }
        }
        
        Log::info('Making Xero API request', [
            'endpoint' => $endpoint,
            'method' => $method,
            'token_id' => $xeroToken->id,
            'tenant_id' => $xeroToken->tenant_id,
            'customer_id' => $customerId,
            'workspace_id' => $workspaceId,
        ]);
        
        $defaultOptions = [
            'headers' => [
                'Authorization' => 'Bearer ' . $accessToken,
                'Accept' => 'application/json',
                'Content-Type' => 'application/json',
                'Xero-tenant-id' => $xeroToken->tenant_id,
            ],
        ];
        $options = array_merge_recursive($defaultOptions, $options);
        try {
            $response = $this->client->request($method, $this->baseUrl . $endpoint, $options);
            return json_decode($response->getBody()->getContents(), true);
        } catch (\GuzzleHttp\Exception\ClientException $e) {
            $response = $e->getResponse();
            $errorBody = $response ? $response->getBody()->getContents() : 'No response body';
            // Try to parse as JSON first, then XML
            $errorData = json_decode($errorBody, true);
            $errorMessage = $e->getMessage();
            if (!$errorData) {
                // Try parsing as XML (Xero sometimes returns XML errors)
                if (strpos($errorBody, '<') !== false) {
                    try {
                        $xml = simplexml_load_string($errorBody);
                        if ($xml) {
                            $errorMessage = (string)($xml->Message ?? $xml->Type ?? $errorBody);
                            $errorData = json_decode(json_encode($xml), true);
                        }
                    } catch (\Exception $xmlError) {
                        // If XML parsing fails, use raw body
                        $errorMessage = $errorBody;
                    }
                } else {
                    $errorMessage = $errorBody;
                }
            } else {
                $errorMessage = isset($errorData['Message']) 
                    ? $errorData['Message'] 
                    : (isset($errorData['error_description']) 
                        ? $errorData['error_description'] 
                        : (isset($errorData['error']) 
                            ? $errorData['error'] 
                            : $errorBody));
            }
            
            // Check if it's a 401 Unauthorized error
            $statusCode = $response ? $response->getStatusCode() : null;
            if ($statusCode == 401) {
                // Check if tenant_id is missing or invalid
                if (!$xeroToken->tenant_id) {
                    $errorMessage = 'Xero authentication failed: tenant_id is missing. Please disconnect and reconnect your Xero account from settings.';
                } else {
                    // Check if it's a scope issue (payroll.timesheets might be missing)
                    if (strpos($endpoint, '/payroll.xro/1.0/Timesheets') !== false) {
                        $errorMessage = 'Xero authentication failed: Your Xero connection is missing the required "payroll.timesheets" scope. This is required to sync timesheets. Please disconnect your Xero account and reconnect it from settings. When reconnecting, ensure you grant all requested permissions, including timesheets access.';
                    } else {
                        $errorMessage = 'Xero authentication failed: The access token may be expired or invalid, or the organization may not have the required permissions. Please disconnect and reconnect your Xero account from settings.';
                    }
                }
                Log::error('Xero API 401 Unauthorized error', [
                    'endpoint' => $endpoint,
                    'method' => $method,
                    'token_id' => $xeroToken->id,
                    'tenant_id' => $xeroToken->tenant_id,
                    'customer_id' => $customerId,
                    'workspace_id' => $workspaceId,
                    'error_body' => $errorBody,
                    'note' => 'This error typically occurs when the token was created without the required scopes. The user must disconnect and reconnect their Xero account to get a new token with updated scopes.',
                ]);
                throw new \Exception($errorMessage);
            }
            
            // Extract validation errors from Xero response if available
            $validationErrors = [];
            // Check for Timesheets validation errors
            if (isset($errorData['Timesheets']) && is_array($errorData['Timesheets'])) {
                foreach ($errorData['Timesheets'] as $timesheetError) {
                    if (isset($timesheetError['ValidationErrors']) && is_array($timesheetError['ValidationErrors'])) {
                        foreach ($timesheetError['ValidationErrors'] as $validationError) {
                            $validationErrors[] = $validationError['Message'] ?? 'Unknown validation error';
                        }
                    }
                }
            }
            // Check for Employees validation errors
            if (isset($errorData['Employees']) && is_array($errorData['Employees'])) {
                foreach ($errorData['Employees'] as $employeeError) {
                    if (isset($employeeError['ValidationErrors']) && is_array($employeeError['ValidationErrors'])) {
                        foreach ($employeeError['ValidationErrors'] as $validationError) {
                            $validationErrors[] = $validationError['Message'] ?? 'Unknown validation error';
                        }
                    }
                    // Also check HomeAddress validation errors
                    if (isset($employeeError['HomeAddress']['ValidationErrors']) && is_array($employeeError['HomeAddress']['ValidationErrors'])) {
                        foreach ($employeeError['HomeAddress']['ValidationErrors'] as $validationError) {
                            $validationErrors[] = 'HomeAddress: ' . ($validationError['Message'] ?? 'Unknown validation error');
                        }
                    }
                }
            }
            // Build detailed error message
            $detailedErrorMessage = $errorMessage;
            if (!empty($validationErrors)) {
                $detailedErrorMessage .= ' | Validation Errors: ' . implode(', ', $validationErrors);
            }
            Log::error('Xero API request error', [
                'status_code' => $response ? $response->getStatusCode() : null,
                'endpoint' => $endpoint,
                'method' => $method,
                'error_body' => $errorBody,
                'error_data' => $errorData,
                'validation_errors' => $validationErrors,
                'full_error_message' => $detailedErrorMessage,
            ]);
            throw new \Exception('Xero API Error: ' . $detailedErrorMessage);
        } catch (\Exception $e) {
            Log::error('Xero API request error: ' . $e->getMessage(), [
                'endpoint' => $endpoint,
            ]);
            throw $e;
        }
    }

    /**
     * Parse Xero date format /Date(timestamp+offset)/ to Carbon
     */
    public function parseXeroDateToCarbon(?string $xeroDate): ?Carbon
    {
        if (!$xeroDate) return null;
        // Matches: /Date(1765756800000+0000)/
        if (preg_match('/\/Date\((\d+)([+-]\d{4})?\)\//', $xeroDate, $m)) {
            $ms = (int) $m[1];
            return Carbon::createFromTimestampUTC((int) ($ms / 1000))->startOfDay();
        }
        return null;
    }

    /**
     * Calculate current period dates from PayrollCalendar
     * Uses PaymentDate to determine the current pay period end date
     * Returns start_date and end_date for the current pay period
     */
    public function calculateCurrentPeriodDates($payrollCalendar)
    {
        try {
            $calendarType = strtoupper($payrollCalendar['CalendarType'] ?? '');
            $startDateStr = $payrollCalendar['StartDate'] ?? null;
            if (!$calendarType) {
                Log::error('PayrollCalendar missing CalendarType', ['calendar' => $payrollCalendar]);
                return null;
            }
            $today = Carbon::today('UTC')->startOfDay();
            // Parse Xero /Date(…+0000)/ to Carbon (UTC)
            $calendarStartDate = $this->parseXeroDateToCarbon($startDateStr) ?? $today->copy();
            $calendarStartDate = $calendarStartDate->startOfDay();
            // Try to use PaymentDate to determine the period end
            // PaymentDate typically represents when payment is made for the period that just ended
            $paymentDateStr = $payrollCalendar['PaymentDate'] ?? null;
            $paymentDate = null;
            if ($paymentDateStr) {
                $paymentDate = $this->parseXeroDateToCarbon($paymentDateStr);
            }
            // Parse ReferenceDate if available (used for monthly calendars)
            $referenceDateStr = $payrollCalendar['ReferenceDate'] ?? null;
            $referenceDate = null;
            if ($referenceDateStr) {
                $referenceDate = $this->parseXeroDateToCarbon($referenceDateStr);
            }
            switch ($calendarType) {
                case 'WEEKLY': {
                    $startDayOfWeek = $calendarStartDate->dayOfWeek;
                    if ($paymentDate) {
                        // PaymentDate is when payment is made - the period typically ends the day before payment
                        // For weekly: if payment is Dec 22 (Monday), period should end Dec 21 (Sunday)
                        // But we need to ensure the period aligns with calendar StartDate day of week
                        // Calculate how many weeks from calendar start to payment date
                        $weeksFromStart = $calendarStartDate->diffInWeeks($paymentDate, false);
                        // The period that ends before this payment date
                        // Period end should be (startDayOfWeek + 6) mod 7, which is 6 days after start
                        $targetPeriodEnd = $calendarStartDate->copy()->addWeeks($weeksFromStart)->addDays(6);
                        // If target period end is on or after payment date, go back one week
                        if ($targetPeriodEnd->gte($paymentDate)) {
                            $targetPeriodEnd = $targetPeriodEnd->copy()->subWeek();
                        }
                        $periodEnd = $targetPeriodEnd;
                        $periodStart = $periodEnd->copy()->subDays(6);
                        // Verify alignment
                        if ($periodStart->dayOfWeek != $startDayOfWeek) {
                            // Adjust to ensure alignment
                            $daysToAdjust = ($periodStart->dayOfWeek - $startDayOfWeek + 7) % 7;
                            $periodStart->subDays($daysToAdjust);
                            $periodEnd = $periodStart->copy()->addDays(6);
                        }
                        Log::info('Using PaymentDate to calculate period', [
                            'payment_date' => $paymentDate->format('Y-m-d'),
                            'calendar_start' => $calendarStartDate->format('Y-m-d'),
                            'weeks_from_start' => $weeksFromStart,
                            'calculated_period_start' => $periodStart->format('Y-m-d'),
                            'calculated_period_end' => $periodEnd->format('Y-m-d'),
                        ]);
                    } else {
                        // No PaymentDate - calculate from StartDate
                        $daysSinceStart = $calendarStartDate->diffInDays($today, false);
                        $weeksOffset = $daysSinceStart >= 0 ? intdiv($daysSinceStart, 7) : 0;
                        $periodStart = $calendarStartDate->copy()->addWeeks($weeksOffset);
                        $periodEnd = $periodStart->copy()->addDays(6);
                        // If period end is in the future, use the previous completed period
                        if ($periodEnd->gt($today)) {
                            $periodStart = $periodStart->copy()->subWeek();
                            $periodEnd = $periodStart->copy()->addDays(6);
                        }
                    }
                    // Final validation: ensure period end is not too far in the future
                    // Xero typically accepts periods ending today or in the near future (within the pay period)
                    if ($periodEnd->gt($today) && $periodEnd->diffInDays($today) > 3) {
                        // Period end is more than 3 days in future, use previous period
                        $periodStart = $periodStart->copy()->subWeek();
                        $periodEnd = $periodStart->copy()->addDays(6);
                    }
                    break;
                }
                case 'FORTNIGHTLY':
                case 'BIWEEKLY': {
                    // Period length = 14 days
                    $daysSinceStart = $calendarStartDate->diffInDays($today, false);
                    $fortnightOffset = $daysSinceStart >= 0 ? intdiv($daysSinceStart, 14) : 0;
                    $periodStart = $calendarStartDate->copy()->addDays($fortnightOffset * 14);
                    $periodEnd   = $periodStart->copy()->addDays(13);
                    // If period end is in the future, use the previous completed period
                    if ($periodEnd->gt($today)) {
                        $periodStart = $periodStart->copy()->subDays(14);
                        $periodEnd = $periodStart->copy()->addDays(13);
                    }
                    // Ensure period end is not after today
                    if ($periodEnd->gt($today)) {
                        $periodEnd = $today->copy();
                    }
                    break;
                }
                case 'MONTHLY': {
                    // For monthly calendars, Xero uses full calendar months
                    // Period starts on the 1st of the month and ends on the last day of the month
                    // The current period is the month containing today   
                    $currentYear = $today->year;
                    $currentMonth = $today->month;
                    // Period starts on the 1st of the current month
                    $periodStart = Carbon::create($currentYear, $currentMonth, 1, 0, 0, 0, 'UTC');
                    // Period ends on the last day of the current month
                    $periodEnd = $periodStart->copy()->endOfMonth();
                    // For monthly calendars, Xero typically accepts the current month period
                    // even if it hasn't ended yet. However, if PaymentDate suggests the period
                    // has been paid (meaning it's completed), we might need to use that period
                    // But generally, we use the current month
                    // If period end is in the future, it's still valid for monthly calendars
                    // Xero accepts current month periods even if not completed
                    // But ensure period end is not too far in the future (safety check)
                    if ($periodEnd->gt($today) && $periodEnd->diffInDays($today) > 31) {
                        // More than a month in future - use previous month
                        $periodStart = $periodStart->copy()->subMonth();
                        $periodEnd = $periodStart->copy()->endOfMonth();
                    }
                    Log::info('Calculated MONTHLY period dates', [
                        'today' => $today->format('Y-m-d'),
                        'period_start' => $periodStart->format('Y-m-d'),
                        'period_end' => $periodEnd->format('Y-m-d'),
                        'payment_date' => $paymentDate ? $paymentDate->format('Y-m-d') : null,
                        'reference_date' => $referenceDate ? $referenceDate->format('Y-m-d') : null,
                        'calendar_start' => $calendarStartDate->format('Y-m-d'),
                    ]);
                    break;
                }
                default: {
                    Log::warning('Unsupported CalendarType in calculateCurrentPeriodDates', [
                        'calendar_type' => $calendarType
                    ]);
                    return null;
                }
            }
            Log::info('Calculated period dates from PayrollCalendar', [
                'calendar_type' => $calendarType,
                'calendar_start' => $calendarStartDate->format('Y-m-d'),
                'today' => $today->format('Y-m-d'),
                'period_start' => $periodStart->format('Y-m-d'),
                'period_end' => $periodEnd->format('Y-m-d'),
            ]);
            return [
                'start_date' => $periodStart->format('Y-m-d'),
                'end_date'   => $periodEnd->format('Y-m-d'),
            ];
        } catch (\Exception $e) {
            Log::error('Error calculating current period dates', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
            return null;
        }
    }

    /**
     * Try to fetch existing timesheets from Xero to find valid period dates
     * This helps identify what periods Xero actually accepts
     * Returns the period that contains today, or the most recent completed period
     * Also returns the original Xero date strings to preserve exact formatting
     */
    public function getValidPeriodFromXeroTimesheets($xeroEmpId, $payrollCalendarId)
    {
        try {
            // Try to fetch recent timesheets for this employee
            $filter = "EmployeeID=={$xeroEmpId}";
            // Get timesheets from the last 60 days to find current/previous periods
            $endDate = Carbon::today()->format('Y-m-d');
            $startDate = Carbon::today()->subDays(60)->format('Y-m-d');
            Log::info('Fetching existing timesheets from Xero to find valid periods', [
                'xero_emp_id' => $xeroEmpId,
                'payroll_calendar_id' => $payrollCalendarId,
                'start_date' => $startDate,
                'end_date' => $endDate,
            ]);
            $response = $this->makeRequest('GET', '/payroll.xro/1.0/Timesheets', [
                'query' => [
                    'filter' => $filter,
                    'startDate' => $startDate,
                    'endDate' => $endDate,
                ]
            ]);
            $today = Carbon::today();
            $bestPeriod = null;
            $bestPeriodEnd = null;
            $bestPeriodXeroDates = null;
            // Extract period dates from existing timesheets
            if (isset($response['Timesheets']) && is_array($response['Timesheets']) && count($response['Timesheets']) > 0) {
                foreach ($response['Timesheets'] as $timesheet) {
                    $startDateStr = $timesheet['StartDate'] ?? null;
                    $endDateStr = $timesheet['EndDate'] ?? null;
                    if ($startDateStr && $endDateStr) {
                        $periodStart = $this->parseXeroDateToCarbon($startDateStr);
                        $periodEnd = $this->parseXeroDateToCarbon($endDateStr);
                        if ($periodStart && $periodEnd) {
                            // Check if this period contains today
                            if ($periodStart->lte($today) && $periodEnd->gte($today)) {
                                // This period contains today - use it with original Xero date strings
                                Log::info('Found period containing today from existing Xero timesheet', [
                                    'start_date' => $periodStart->format('Y-m-d'),
                                    'end_date' => $periodEnd->format('Y-m-d'),
                                    'xero_start_date' => $startDateStr,
                                    'xero_end_date' => $endDateStr,
                                ]);
                                return [
                                    'start_date' => $periodStart->format('Y-m-d'),
                                    'end_date' => $periodEnd->format('Y-m-d'),
                                    'xero_start_date' => $startDateStr, // Preserve original Xero format
                                    'xero_end_date' => $endDateStr, // Preserve original Xero format
                                ];
                            }
                            // Track the most recent completed period (end date <= today)
                            if ($periodEnd->lte($today) && (!$bestPeriodEnd || $periodEnd->gt($bestPeriodEnd))) {
                                $bestPeriod = [
                                    'start_date' => $periodStart->format('Y-m-d'),
                                    'end_date' => $periodEnd->format('Y-m-d'),
                                ];
                                $bestPeriodXeroDates = [
                                    'xero_start_date' => $startDateStr,
                                    'xero_end_date' => $endDateStr,
                                ];
                                $bestPeriodEnd = $periodEnd;
                            }
                        }
                    }
                }
                // If we found a completed period but not one containing today, use the most recent one
                if ($bestPeriod && $bestPeriodXeroDates) {
                    Log::info('Using most recent completed period from existing Xero timesheet', [
                        'start_date' => $bestPeriod['start_date'],
                        'end_date' => $bestPeriod['end_date'],
                    ]);
                    return array_merge($bestPeriod, $bestPeriodXeroDates);
                }
            }
            return null;
        } catch (\Exception $e) {
            Log::warning('Failed to fetch existing timesheets from Xero', [
                'xero_emp_id' => $xeroEmpId,
                'error' => $e->getMessage(),
            ]);
            return null;
        }
    }

    /**
     * Get daily hours from attendance records for a date range
     * Returns an array where each element represents hours for a day
     */
    public function getDailyHoursFromAttendance($employeeId, $startDate, $endDate, $customerId, $workspaceId)
    {
        $dailyHours = [];
        // Create date range
        $currentDate = $startDate->copy();
        $endDateCopy = $endDate->copy();
        // Get all attendance records for the date range
        $attendances = EmployeeAttendance::where('employee_id', $employeeId)
            ->where('customer_id', $customerId)
            ->where('workspace_id', $workspaceId)
            ->whereBetween('date', [$startDate->format('Y-m-d'), $endDateCopy->format('Y-m-d')])
            ->where('status', 1) // Only active/approved attendance
            ->whereNotNull('check_in')
            ->whereNotNull('check_out')
            ->get()
            ->keyBy(function ($attendance) {
                return Carbon::parse($attendance->date)->format('Y-m-d');
            });
        // Loop through each day in the date range
        while ($currentDate->lte($endDateCopy)) {
            $dateKey = $currentDate->format('Y-m-d');
            // Get attendance for this specific date
            $attendance = $attendances->get($dateKey);
            if ($attendance && $attendance->working_hours) {
                // Convert minutes to hours and round to 2 decimal places
                $hours = round($attendance->working_hours / 60, 2);
                $dailyHours[] = $hours;
            } else {
                // No attendance for this day, add 0 hours
                $dailyHours[] = 0.00;
            }
            // Move to next day
            $currentDate->addDay();
        }
        return $dailyHours;
    }
}
