<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class IncidentReport extends BaseModel
{
    use HasFactory;

    protected $fillable = [
        'title',
        'customer_id',
        'workspace_id',
        'is_victim',
        'document_number',
        'authorised_by',
        'revision_number',
        'issue_date',
        'site_id',
        'revision_date',
        'parent_number',
        'investigation_number',
        'office_number',
        'project_id',
        'project_name',
        'project_number',
        'report_compiled_by',
        'position_id',
        'date',
        'incident_date',
        'incident_time',
        'date_incident_reported',
        'time_incident_reported',
        'is_notified',
        'is_notified_to',
        'involved_persons',
        'is_extra_questions',
        'is_plant_involved',
        'is_correct_action_required',
        'plant_name',
        'is_injury_management',
        'is_final_injury_classification',
        'is_final_incident_classification',
        'lti_or_rwi',
        'incident_response',
        'del',
        'process',
        'step_no',
        'is_published',
        'is_questions',
        'report_method',
    ];
    
    protected $hidden = [
        'customer_id',
        'workspace_id',
    ];
    public function project()
    {
        return $this->belongsTo(Project::class, 'project_id');
    }

    // 2️⃣ Site Related
    public function site()
    {
        return $this->belongsTo(Sites::class, 'site_id');
    }

    // 3️⃣ Authorised By (Employee Detail)
    public function authorisedBy()
    {
        return $this->belongsTo(EmpPersonalDetails::class, 'authorised_by', 'emp_id');
    }
    public function sitesData()
    {
        return $this->belongsTo(Sites::class, 'site_id', 'id');
    }
    public function authorisedByCompany()
    {
        return $this->belongsTo(EmpCompanyDetails::class, 'authorised_by', 'id');
    }

    // 4️⃣ Reported To (Employees)
    public function reportedTos()
    {
        return $this->hasMany(IncidentReportedTo::class, 'incident_report_id');
    }

    public function reportedToDetails()
    {
        return $this->hasOneThrough(EmpPersonalDetails::class, IncidentReportedTo::class, 'incident_report_id', 'emp_id', 'id', 'reported_to');
    }

    // 5️⃣ Reported By (Employees)
    public function reportedBys()
    {
        return $this->hasMany(IncidentReportedBy::class, 'incident_report_id');
    }
    public function customer()
    {
        return $this->belongsTo(User::class, 'customer_id', 'id')
            ->where('user_type', config('constants.user_types.customer'))
            ->select('id as emp_id', 'name as first_name');  // Fixed: using 'name' column, aliasing as 'first_name'
    }



    public function reportedByDetails()
    {
        return $this->hasOneThrough(EmpPersonalDetails::class, IncidentReportedBy::class, 'incident_report_id', 'emp_id', 'id', 'reported_by');
    }

    // 6️⃣ Incident Classification Reports
    public function classifications()
    {
        return $this->hasMany(IncidentClassificationReport::class, 'incident_report_id');
    }

    public function injuryClassifications()
    {
        return $this->hasManyThrough(
            IncidentInjuryClassification::class,
            IncidentClassificationReport::class,
            'incident_report_id', // Foreign key on the IncidentClassificationReport table
            'id', // Primary key on the IncidentInjuryClassification table
            'id', // Local key on the IncidentReport table
            'injury_classification_id' // Foreign key on the IncidentClassificationReport table
        )->select([
            'incident_injury_classifications.id as injury_classification_id',
            'incident_injury_classifications.title as injury_classification_title',
            'incident_injury_classifications.description as injury_classification_description'
        ]);
    }

    // 7️⃣ Notified To (Incident Notification)
    public function notifiedTos()
    {
        return $this->hasMany(IncidentNotifiedTo::class, 'incident_report_id');
    }

    // 8️⃣ Involved Persons
    public function involvedPersons()
    {
        return $this->hasOne(IncidentInvolvedPersonDetail::class, 'incident_report_id');
    }
    public function involved_persons_details() {
        return $this->hasMany(IncidentInvolvedPersonDetail::class, 'incident_report_id')
                    ->with('employmentType:id,title')
                    ->with('employee:id,first_name,last_name');
    }

    public function involvedTrainings()
    {
        return $this->belongsToMany(
            Training::class, // Target model
            'incident_involved_person_trainings', // Pivot table
            'incident_report_id', // Foreign key in pivot table referencing IncidentReport
            'training_id' // Foreign key in pivot table referencing Training
        );
    }

    public function injuryTypes()
    {
        return $this->hasMany(IncidentEmployeeInjuryType::class, 'incident_report_id');
    }

    // 9️⃣ Employer Details
    public function employerDetails()
    {
        return $this->hasOne(IncidentEmployerDetail::class, 'incident_report_id');
    }

    // 🔟 Incident Questions and Answers
    public function questionAnswers()
    {
        return $this->hasMany(IncidentReportQuestionAnswer::class, 'incident_report_id');
    }

    public function questions()
    {
        return $this->hasManyThrough(
            IncidentReportQuestion::class,
            IncidentReportQuestionAnswer::class,
            'incident_report_id', // Foreign key in `incident_report_question_answers`
            'id', // Primary key in `incident_report_questions`
            'id', // Primary key in `incident_reports`
            'question_id' // Foreign key in `incident_report_question_answers`
        )->with('answer'); // Load the answer for each question
    }

    // 1️⃣1️⃣ Incident Meta Data
    public function incidentReportMetas()
    {
        return $this->hasOne(IncidentReportMeta::class, 'incident_report_id');
    }

    // 1️⃣2️⃣ Injury Management
    public function injuryManagements()
    {
        return $this->hasOne(IncidentInjuryManagement::class, 'incident_report_id');
    }

    // 1️⃣3️⃣ Final Classifications
    public function finalClassifications()
    {
        return $this->hasOne(IncidentFinalClassification::class, 'incident_report_id');
    }

    // 1️⃣4️⃣ Signoffs (Employee Signatures)
    public function signoffs()
    {
        return $this->hasMany(IncidentSignoff::class, 'incident_report_id');
    }

    // 1️⃣5️⃣ Risk Category Details
    public function incidentRiskCategoryDetails()
    {
        return $this->hasOne(IncidentRiskCategoryDetail::class, 'incident_report_id');
    }

    // 1️⃣6️⃣ Witnesses
    public function incidentWitnesses()
    {
        return $this->hasMany(IncidentWitnes::class, 'incident_report_id');
    }

    // 1️⃣7️⃣ Incident Documents
    public function incidentDocuments()
    {
        return $this->hasMany(IncidentDocument::class, 'incident_report_id');
    }

    // 1️⃣8️⃣ Supervisor Details (For Site)
    public function supervisor()
    {
        return $this->hasOneThrough(
            EmpPersonalDetails::class,
            Sites::class,
            'id', // Foreign key in Site table
            'emp_id', // Foreign key in Employee table
            'site_id', // Local key in IncidentReport table
            'supervisor_id' // Foreign key in Site table
        );
    }

    // 1️⃣9️⃣ Foreman Details (For Site)
    public function forman()
    {
        return $this->hasOneThrough(
            EmpPersonalDetails::class,
            Sites::class,
            'id',
            'emp_id',
            'site_id',
            'forman_id'
        );
    }

    public function incidentClassificationReports()
    {
        return $this->hasMany(IncidentClassificationReport::class, 'incident_report_id');
    }

    public function notifiableClassifications()
    {
        return $this->belongsToMany(
            IncidentNotifiableClassification::class,
            'incident_report_notifiable_classifications', // Pivot table
            'incident_report_id', // Foreign key in pivot table referencing IncidentReport
            'notifiable_classification_id' // Foreign key in pivot table referencing IncidentNotifiableClassification
        );
    }

    public function images()
    {
        return $this->hasMany(IncidentImage::class, 'incident_report_id');
    }
}
