<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class LeaveRequest extends BaseModel
{
    use HasFactory;
    protected $fillable = [
        'employee_id',
        'leave_package_id',
        'from',
        'to',
        'reason',
        'applied_by',
        'status',
        'approved_by',
        'full_day',
        'end_time',
        'start_time',
        'customer_id',
        'workspace_id',
    ];

    protected $hidden = [
        'customer_id',
        'workspace_id',
    ];

    public function leavepackage()
    {
        return $this->belongsTo(LeavePackage::class, 'leave_package_id');
    }

    public function emppersonaldetails()
    {
        return $this->belongsTo(EmpPersonalDetails::class, 'employee_id', 'emp_id');
    }

    public function empDetails()
    {
        return $this->hasOne(EmpPersonalDetails::class, 'emp_id', 'employee_id');
    }

    public function empCompanyDetails()
    {
        return $this->belongsTo(EmpCompanyDetails::class, 'employee_id');
    }
    public function leave_employee_details()
    {
        return $this->belongsTo(EmpPersonalDetails::class, 'employee_id', 'emp_id')->select('emp_id', 'first_name', 'middle_name', 'last_name', 'image');
    }
    public function leave_employee_company_details()
    {
        return $this->belongsTo(EmpCompanyDetails::class, 'employee_id', 'id')->select('id', 'employee_email', 'access_role');
    }

    /**
     * Get the 'from' date in server format for database queries
     * This ensures consistent date format regardless of user preferences
     */
    public function getFromDateForQuery()
    {
        if ($this->from) {
            // Parse the date using Carbon and return in Y-m-d format
            try {
                return \Carbon\Carbon::parse($this->from)->format('Y-m-d');
            } catch (\Exception $e) {
                // If parsing fails, return the original value
                return $this->from;
            }
        }
        return null;
    }

    /**
     * Get the 'to' date in server format for database queries
     * This ensures consistent date format regardless of user preferences
     */
    public function getToDateForQuery()
    {
        if ($this->to) {
            // Parse the date using Carbon and return in Y-m-d format
            try {
                return \Carbon\Carbon::parse($this->to)->format('Y-m-d');
            } catch (\Exception $e) {
                // If parsing fails, return the original value
                return $this->to;
            }
        }
        return null;
    }

    /**
     * Scope to get leave requests for a specific date range
     * This method handles date formatting properly for database queries
     */
    public function scopeForDateRange($query, $startDate, $endDate)
    {
        return $query->where(function ($q) use ($startDate, $endDate) {
            $q->where('from', '<=', $startDate)
                ->where('to', '>=', $endDate);
        });
    }

    /**
     * Scope to get leave requests for today
     * This method uses server date to ensure consistency
     */
    public function scopeForToday($query)
    {
        $today = self::generateServerDate();
        return $query->where(function ($q) use ($today) {
            $q->where('from', '<=', $today)
                ->where('to', '>=', $today);
        });
    }
}
