<?php

namespace App\Http\Controllers\Subcontractor;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Tender;
use App\Models\TenderLineItem;
use App\Models\TenderParticipant;
use App\Models\TenderLineItemsBid;
use App\Models\TenderAttachment;
use App\Models\TenderRfi;
use App\Models\User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class SubcontractorTenderController extends Controller
{
    /**
     * Get all invited tenders for authenticated subcontractor
     */
    public function getInvitedTenders(Request $request)
    {
        $user = Auth::user();
        if (!$user) {
            return $this->message('Unauthorized', 401);
        }
        $query = Tender::with(['project', 'lineItems',])
            ->whereHas('participants', function ($q) use ($user) {
                $q->where('user_id', $user->id)
                  ->where('invite_status', 'invited');
            })
            ->where('is_published', 1)
            ->where('status', 'open');

        // Filter by status if provided
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }
        $tenders = $query->orderBy('created_at', 'desc')->get();
        // Add assignment status for each tender
        $tenders = $tenders->map(function ($tender) use ($user) {
            $participant = $tender->participants->where('user_id', $user->id)->first();
            $tender->is_assigned = $participant ? $participant->is_assigned : false;
            $tender->assigned_at = $participant ? $participant->assigned_at : null;
            // Check if user has submitted a bid
            $hasBid = TenderLineItemsBid::where('tender_id', $tender->id)
                ->where('user_id', $user->id)
                ->where('status', 'submitted')
                ->exists();
            $tender->has_submitted_bid = $hasBid;
            return $tender;
        });
        return $this->success($tenders, 'Invited tenders retrieved successfully');
    }

    /**
     * Get invited tender by ID with documents/attachments
     */
    public function getInvitedTenderById($id)
    {
        $user = Auth::user();
        if (!$user) {
            return $this->message('Unauthorized', 401);
        }

        // Get tender with relations
        $tender = Tender::with(['project', 'lineItems', 'participants'])
            ->whereHas('participants', function ($q) use ($user) {
                $q->where('user_id', $user->id)
                  ->where('invite_status', 'invited');
            })
            ->where('id', $id)
            ->where('is_published', 1)
            ->where('status', 'open')
            ->first();

        if (!$tender) {
            return $this->message('Tender not found or you are not invited', 404);
        }

        // Get participant details
        $participant = $tender->participants->where('user_id', $user->id)->first();
        $tender->is_assigned = $participant ? $participant->is_assigned : false;
        $tender->assigned_at = $participant ? $participant->assigned_at : null;

        // Check if user has submitted a bid
        $hasBid = TenderLineItemsBid::where('tender_id', $tender->id)
            ->where('user_id', $user->id)
            ->where('status', 'submitted')
            ->exists();
        $tender->has_submitted_bid = $hasBid;

        // Get admin uploaded documents/attachments (entity_type = 'customer')
        $documents = TenderAttachment::where('entity_type', 'customer')
            ->where('entity_id', $tender->id)
            ->with('uploadedBy')
            ->orderBy('created_at', 'desc')
            ->get();
        
        $tender->documents = $documents;

        return $this->success($tender, 'Invited tender retrieved successfully');
    }

    /**
     * Add/Update bid prices for line items
     */
    public function addBidPrices(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'tender_id' => 'required|integer|exists:tenders,id',
            'line_items' => 'required|array|min:1',
            'line_items.*.line_item_id' => 'required|integer|exists:tender_line_items,id',
            'line_items.*.bid_price' => 'required|numeric|min:0',
            'total_base_bid' => 'nullable|numeric|min:0',
            'exclusions_clarifications' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }

        $user = Auth::user();
        if (!$user) {
            return $this->message('Unauthorized', 401);
        }

        // Verify tender is published and open
        $tender = Tender::find($request->tender_id);
        if (!$tender || $tender->is_published != 1 || $tender->status != 'open') {
            return $this->message('Tender not found or not available for bidding', 404);
        }

        // Verify user is invited
        $participant = TenderParticipant::where('tender_id', $tender->id)
            ->where('user_id', $user->id)
            ->where('invite_status', 'invited')
            ->first();
        
        if (!$participant) {
            return $this->message('You are not invited to this tender', 403);
        }

        DB::beginTransaction();
        try {
            $ids = $this->getCustomerAndWorkspaceIds();
            if (!$ids) {
                return $this->message('Unable to determine customer or workspace', 400);
            }

            // Calculate total if not provided
            $totalBaseBid = $request->total_base_bid ?? array_sum(array_column($request->line_items, 'bid_price'));

            // Update or create bid prices for each line item
            foreach ($request->line_items as $item) {
                TenderLineItemsBid::updateOrCreate(
                    [
                        'tender_id' => $tender->id,
                        'tender_line_items_id' => $item['line_item_id'],
                        'user_id' => $user->id,
                    ],
                    [
                        'bid_price' => $item['bid_price'],
                        'total_base_bid' => $totalBaseBid,
                        'exclusions_clarifications' => $request->exclusions_clarifications,
                        'status' => 'draft',
                        'customer_id' => $ids['customer_id'],
                        'workspace_id' => $ids['workspace_id'],
                    ]
                );
            }

            DB::commit();
            return $this->success(null, 'Bid prices saved successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error saving bid prices: ' . $e->getMessage());
            return $this->message('Failed to save bid prices', 500);
        }
    }

    /**
     * Submit bid with attachments
     */
    public function submitBid(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'tender_id' => 'required|integer|exists:tenders,id',
            'attachments' => 'nullable|array',
            'attachments.*' => 'file|mimes:pdf,doc,docx,xlsx,jpg,jpeg,png|max:10240',
            'attachments.*.title' => 'required_with:attachments.*|string|max:255',
        ]);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        $user = Auth::user();
        if (!$user) {
            return $this->message('Unauthorized', 401);
        }
        $tender = Tender::find($request->tender_id);
        if (!$tender || $tender->is_published != 1 || $tender->status != 'open') {
            return $this->message('Tender not found or not available for bidding', 404);
        }
        // Check if tender is closed
        if ($tender->closing_date && \Carbon\Carbon::parse($tender->closing_date)->isPast()) {
            if (!$tender->allow_late_submissions) {
                return $this->message('Tender has closed and late submissions are not allowed', 400);
            }
        }
        // Verify user has bid prices
        $bidPrices = TenderLineItemsBid::where('tender_id', $tender->id)
            ->where('user_id', $user->id)
            ->where('status', 'draft')
            ->get();

        if ($bidPrices->isEmpty()) {
            return $this->message('Please add bid prices before submitting', 400);
        }

        DB::beginTransaction();
        try {
            $ids = $this->getCustomerAndWorkspaceIds();
            if (!$ids) {
                return $this->message('Unable to determine customer or workspace', 400);
            }
            $isLateSubmission = $tender->closing_date && \Carbon\Carbon::parse($tender->closing_date)->isPast();
            $totalBaseBid = $bidPrices->sum('bid_price');
            $exclusionsClarifications = $bidPrices->first()->exclusions_clarifications;
            // Update all bid prices to submitted status
            TenderLineItemsBid::where('tender_id', $tender->id)
                ->where('user_id', $user->id)
                ->where('status', 'draft')
                ->update([
                    'status' => 'submitted',
                    'total_base_bid' => $totalBaseBid,
                    'exclusions_clarifications' => $exclusionsClarifications,
                    'is_late_submission' => $isLateSubmission,
                    'submitted_at' => now(),
                ]);
            // Handle attachments
            if ($request->hasFile('attachments')) {
                $uploadPath = 'tender_bid_attachments';
                $fullUploadPath = public_path($uploadPath);
                if (!file_exists($fullUploadPath)) {
                    mkdir($fullUploadPath, 0755, true);
                }
                foreach ($request->file('attachments') as $index => $file) {
                    if ($file->isValid()) {
                        $filename = time() . '_' . uniqid() . '_' . $file->getClientOriginalName();
                        $file->move($fullUploadPath, $filename);
                        $filePath = $uploadPath . '/' . $filename;

                        $title = $request->input("attachments.{$index}.title") ?? $file->getClientOriginalName();

                        TenderAttachment::create([
                            'entity_type' => 'tender_bid',
                            'entity_id' => $tender->id . '_' . $user->id, // Composite ID for bid
                            'title' => $title,
                            'file_name' => $filename,
                            'file_path' => $filePath,
                            'uploaded_by' => $user->id,
                            'customer_id' => $ids['customer_id'],
                            'workspace_id' => $ids['workspace_id'],
                        ]);
                    }
                }
            }

            DB::commit();
            return $this->success(null, 'Bid submitted successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error submitting bid: ' . $e->getMessage());
            return $this->message('Failed to submit bid', 500);
        }
    }

    /**
     * Get my submitted bids
     */
    public function getMyBids(Request $request)
    {
        $user = Auth::user();
        if (!$user) {
            return $this->message('Unauthorized', 401);
        }

        $query = Tender::with(['project', 'lineItems'])
            ->whereHas('bids', function ($q) use ($user) {
                $q->where('user_id', $user->id)
                  ->where('status', 'submitted');
            });

        if ($request->filled('tender_id')) {
            $query->where('id', $request->tender_id);
        }

        $tenders = $query->orderBy('created_at', 'desc')->get();

        $tenders = $tenders->map(function ($tender) use ($user) {
            $bid = TenderLineItemsBid::where('tender_id', $tender->id)
                ->where('user_id', $user->id)
                ->where('status', 'submitted')
                ->first();
            
            $tender->bid_total = $bid->total_base_bid ?? 0;
            $tender->bid_submitted_at = $bid->submitted_at ?? null;
            $tender->is_late_submission = $bid->is_late_submission ?? false;
            
            // Get attachments
            $tender->attachments = TenderAttachment::where('entity_type', 'tender_bid')
                ->where('entity_id', $tender->id . '_' . $user->id)
                ->get();
            
            return $tender;
        });

        return $this->success($tenders, 'My bids retrieved successfully');
    }

    /**
     * Add RFI question
     */
    public function addRfiQuestion(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'tender_id' => 'required|integer|exists:tenders,id',
            'question' => 'required|string',
        ]);

        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }

        $user = Auth::user();
        if (!$user) {
            return $this->message('Unauthorized', 401);
        }

        $tender = Tender::find($request->tender_id);
        if (!$tender || $tender->is_published != 1) {
            return $this->message('Tender not found', 404);
        }

        // Check RFI cut-off date
        if ($tender->rfi_cut_off_date && \Carbon\Carbon::parse($tender->rfi_cut_off_date)->isPast()) {
            return $this->message('RFI cut-off date has passed', 400);
        }

        $ids = $this->getCustomerAndWorkspaceIds();
        if (!$ids) {
            return $this->message('Unable to determine customer or workspace', 400);
        }

        $rfi = TenderRfi::create([
            'tender_id' => $tender->id,
            'user_id' => $user->id,
            'question' => $request->question,
            'status' => 'pending',
            'customer_id' => $ids['customer_id'],
            'workspace_id' => $ids['workspace_id'],
        ]);

        return $this->success($rfi, 'RFI question submitted successfully');
    }

    /**
     * Get RFIs for a tender (subcontractor sees only their RFIs)
     */
    public function getRfisForTender($tenderId)
    {
        $tender = Tender::find($tenderId);
        if (!$tender) {
            return $this->message('Tender not found', 404);
        }

        $user = Auth::user();
        if (!$user) {
            return $this->message('Unauthorized', 401);
        }

        $rfis = TenderRfi::with('user')
            ->where('tender_id', $tenderId)
            ->where('user_id', $user->id) // Only show subcontractor's RFIs
            ->orderBy('created_at', 'desc')
            ->get();

        return $this->success($rfis, 'RFIs retrieved successfully');
    }
}

