<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Tender;
use App\Models\TenderLineItem;
use App\Models\TenderParticipant;
use App\Models\TenderLineItemsBid;
use App\Models\TenderAttachment;
use App\Models\TenderRfi;
use App\Models\User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\File;

class TenderController extends Controller
{
    /**
     * Step 1: Store/Update basic tender information
     * Fields: project_id, title, closing_date, reference (auto-generated)
     */
    public function step1(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'nullable|integer|exists:tenders,id',
            'project_id' => 'required|integer|exists:projects,id',
            'title' => 'required|string|max:255',
            'closing_date' => 'required|date',
        ]);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        // Get customer_id and workspace_id
        $ids = $this->getCustomerAndWorkspaceIds();
        if (!$ids) {
            return $this->message('Unable to determine customer or workspace', 400);
        }
        $data = [
            'project_id' => $request->project_id,
            'title' => $request->title,
            'closing_date' => $request->closing_date,
            'customer_id' => $ids['customer_id'],
            'workspace_id' => $ids['workspace_id'],
            'status' => 'draft',
        ];
        if ($request->filled('id')) {
            // Update existing tender
            $tender = Tender::find($request->id);
            $tender->update($data);
            $message = 'Tender updated successfully';
        } else {
            // Create new tender (reference will be auto-generated in boot method)
            $tender = Tender::create($data);
            $message = 'Tender created successfully';
        }
        return $this->success($tender, $message);
    }

    /**
     * Step 2: Store/Update scope of work and line items
     * Fields: scope_of_work, line_items (array with description, unit, quantity)
     */
    public function step2(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|integer|exists:tenders,id',
            'scope_of_work' => 'nullable|string',
            'line_items' => 'required|array|min:1',
            'line_items.*.description' => 'required|string',
            'line_items.*.unit' => 'required|string|max:50',
            'line_items.*.quantity' => 'required|integer|min:1',
        ]);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        // Get customer_id and workspace_id and verify tender ownership
        $ids = $this->getCustomerAndWorkspaceIds();
        if (!$ids) {
            return $this->message('Unable to determine customer or workspace', 400);
        }
        $query = Tender::query();
        $query = $this->applyCustomerWorkspaceFilter($query);
        $tender = $query->find($request->id);
        if (!$tender) {
            return $this->message('Tender not found or you do not have access', 404);
        }
        // Update scope of work
        $tender->scope_of_work = $request->scope_of_work;
        $tender->save();
        // Delete existing line items for this tender
        TenderLineItem::where('tender_id', $tender->id)->delete();
        // Create new line items
        $lineItems = [];
        foreach ($request->line_items as $item) {
            $lineItems[] = TenderLineItem::create([
                'tender_id' => $tender->id,
                'description' => $item['description'],
                'unit' => $item['unit'],
                'quantity' => $item['quantity'],
                'customer_id' => $ids['customer_id'],
                'workspace_id' => $ids['workspace_id'],
            ]);
        }
        // Reload tender with line items
        $tender->load('lineItems');
        return $this->success($tender, 'Tender line items saved successfully');
    }

    /**
     * Step 3: Store/Update tender settings
     * Fields: reminder_days_before, allow_late_submissions, allow_bidder_document_uploads
     */
    public function step3(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|integer|exists:tenders,id',
            'reminder_days_before' => 'nullable|integer|min:0',
            'allow_late_submissions' => 'nullable|boolean',
            'allow_bidder_document_uploads' => 'nullable|boolean',
        ]);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }

        // Get customer_id and workspace_id and verify tender ownership
        $ids = $this->getCustomerAndWorkspaceIds();
        if (!$ids) {
            return $this->message('Unable to determine customer or workspace', 400);
        }

        $query = Tender::query();
        $query = $this->applyCustomerWorkspaceFilter($query);
        $tender = $query->find($request->id);

        if (!$tender) {
            return $this->message('Tender not found or you do not have access', 404);
        }
        $data = [];
        if ($request->has('reminder_days_before')) {
            $data['reminder_days_before'] = $request->reminder_days_before;
        }
        if ($request->has('allow_late_submissions')) {
            $data['allow_late_submissions'] = $request->allow_late_submissions;
        }
        if ($request->has('allow_bidder_document_uploads')) {
            $data['allow_bidder_document_uploads'] = $request->allow_bidder_document_uploads;
        }
        $tender->update($data);
        return $this->success($tender, 'Tender settings saved successfully');
    }



    /**
     * Get step data for tender form
     * Accepts step (step1, step2, step3) and tender_id (optional)
     * Returns data according to the step
     */
    public function getStepData(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'step' => 'required|string|in:step0,step1,step2',
            'tender_id' => 'nullable|integer|exists:tenders,id',
        ]);

        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }

        $step = $request->step;
        $tenderId = $request->tender_id;

        // Apply customer workspace filter
        $query = Tender::query();
        $query = $this->applyCustomerWorkspaceFilter($query);

        // If tender_id is provided, verify ownership and get tender
        $tender = null;
        if ($tenderId) {
            $tender = $query->find($tenderId);
            if (!$tender) {
                return $this->message('Tender not found or you do not have access', 404);
            }
        }

        $data = [];

        switch ($step) {
            case 'step0':
                // Step 0: Basic information
                if ($tender) {
                    $data = [
                        'project_id' => $tender->project_id,
                        'title' => $tender->title,
                        'closing_date' => $tender->closing_date,
                        'reference' => $tender->reference,
                        'status' => $tender->status,
                    ];
                } else {
                    // Return empty structure for new tender
                    $data = [
                        'project_id' => null,
                        'title' => null,
                        'closing_date' => null,
                        'reference' => null,
                        'status' => 'draft',
                    ];
                }
                break;

            case 'step1':
                // Step 1: Scope of work and line items
                if ($tender) {
                    $tender->load('lineItems');
                    $data = [
                        'scope_of_work' => $tender->scope_of_work,
                        'line_items' => $tender->lineItems->map(function ($item) {
                            return [
                                'id' => $item->id,
                                'description' => $item->description,
                                'unit' => $item->unit,
                                'quantity' => $item->quantity,
                            ];
                        })->toArray(),
                    ];
                } else {
                    // Return empty structure for new tender
                    $data = [
                        'scope_of_work' => null,
                        'line_items' => [],
                    ];
                }
                break;

            case 'step2':
                // Step 2: Settings
                if ($tender) {
                    $data = [
                        'reminder_days_before' => $tender->reminder_days_before,
                        'allow_late_submissions' => $tender->allow_late_submissions ?? false,
                        'allow_bidder_document_uploads' => $tender->allow_bidder_document_uploads ?? false,
                    ];
                } else {
                    // Return default values for new tender
                    $data = [
                        'reminder_days_before' => null,
                        'allow_late_submissions' => false,
                        'allow_bidder_document_uploads' => false,
                    ];
                }
                break;

            default:
                return $this->message('Invalid step provided', 400);
        }
        $data['step'] = $step;
        $data['tender_id'] = $tenderId;
        return $this->success($data, ucfirst($step) . ' data retrieved successfully');
    }

    /**
     * Get tender by ID
     */
    public function show($id)
    {

        $query = Tender::with(['lineItems', 'project', 'participants', 'rfis']);
        // Apply customer workspace filter
        $query = $this->applyCustomerWorkspaceFilter($query);
        $tender = $query->find($id);
        return $this->success($tender, 'Tender retrieved successfully');
    }

    /**
     * List all tenders
     */
    public function index(Request $request)
    {
        $query = Tender::with(['project', 'lineItems']);
        // Apply customer workspace filter if needed
        $query = $this->applyCustomerWorkspaceFilter($query);
        // Search functionality
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('title', 'like', '%' . $search . '%')
                    ->orWhere('reference', 'like', '%' . $search . '%')
                    ->orWhereHas('project', function ($subquery) use ($search) {
                        $subquery->where('title', 'like', '%' . $search . '%');
                    });
            });
        }
        // Filter by project
        if ($request->filled('project_id')) {
            $query->where('project_id', $request->project_id);
        }
        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }
        $query->orderBy('created_at', 'desc');
        return $this->withCount($query, 'Tenders retrieved successfully');
    }

    /**
     * Delete tender
     */
    public function destroy($id)
    {
        $query = Tender::query();
        // Apply customer workspace filter
        $query = $this->applyCustomerWorkspaceFilter($query);
        $tender = $query->find($id);
        if (!$tender) {
            return $this->message('Tender not found', 404);
        }
        // Delete related records
        TenderLineItem::where('tender_id', $tender->id)->delete();
        $tender->delete();
        return $this->message('Tender deleted successfully', 200);
    }



    /**
     * Invite subcontractors to a tender
     * Accepts tender_id and subcontractor_ids (array or single integer)
     * Creates TenderParticipant records and sends invitation emails
     */
    public function inviteSubcontractors(Request $request)
    {
        // Normalize subcontractor_ids to always be an array
        $subcontractorIds = $request->subcontractor_ids;
        if (!is_array($subcontractorIds)) {
            // If single value, convert to array
            if (is_numeric($subcontractorIds)) {
                $subcontractorIds = [(int)$subcontractorIds];
            } else {
                return $this->message('subcontractor_ids must be an array or integer', 400);
            }
        }
        $validator = Validator::make([
            'tender_id' => $request->tender_id,
            'subcontractor_ids' => $subcontractorIds,
        ], [
            'tender_id' => 'required|integer|exists:tenders,id',
            'subcontractor_ids' => 'required|array|min:1',
            'subcontractor_ids.*' => 'required|integer|exists:users,id',
        ]);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        // Get customer_id and workspace_id and verify tender ownership
        $ids = $this->getCustomerAndWorkspaceIds();
        $query = Tender::query();
        $query = $this->applyCustomerWorkspaceFilter($query);
        $tender = $query->with(['project', 'lineItems'])->find($request->tender_id);
        if (!$tender) {
            return $this->message('Tender not found or you do not have access', 404);
        }
        // Verify all subcontractors belong to this customer/workspace
        $subcontractors = $this->getSubcontractorsForCustomer();
        $validSubcontractorIds = $subcontractors->pluck('id')->toArray();
        $invalidIds = array_diff($subcontractorIds, $validSubcontractorIds);
        if (!empty($invalidIds)) {
            return $this->message('Some subcontractors are not associated with your customer/workspace', 400);
        }
        $invitedCount = 0;
        $emailSentCount = 0;
        $errors = [];
        foreach ($subcontractorIds as $subcontractorId) {
                // Check if participant already exists
                $existingParticipant = TenderParticipant::where('tender_id', $tender->id)
                    ->where('user_id', $subcontractorId)
                    ->first();
                if ($existingParticipant) {
                    // Update invite status if already exists
                    $existingParticipant->update([
                        'invite_status' => 'invited',
                    ]);
                    $invitedCount++;
                } else {
                    // Create new participant record
                    TenderParticipant::create([
                        'tender_id' => $tender->id,
                        'user_id' => $subcontractorId,
                        'invite_status' => 'invited',
                        'customer_id' => $ids['customer_id'],
                        'workspace_id' => $ids['workspace_id'],
                    ]);
                    $invitedCount++;
                }
                // Get subcontractor details
                $subcontractor = User::find($subcontractorId);
                if (!$subcontractor || !filter_var($subcontractor->email, FILTER_VALIDATE_EMAIL)) {
                    $errors[] = "Invalid email for subcontractor ID: {$subcontractorId}";
                    continue;
                }
                // Format closing date
                $closingDate = $tender->closing_date ? \Carbon\Carbon::parse($tender->closing_date)->format('d/m/Y') : 'N/A';
                // Get project name
                $projectName = $tender->project ? $tender->project->title : 'N/A';
                // Prepare email subject
                $subject = "Tender Invitation - {$tender->title}";
                $subject .= " | " . env('APP_NAME', 'WMS');
                // Prepare email content
                $emailContent = view('Emails.tender-invitation', [
                    'subcontractor_name' => $subcontractor->name ?? 'there',
                    'tender_title' => $tender->title,
                    'tender_reference' => $tender->reference,
                    'project_name' => $projectName,
                    'closing_date' => $closingDate,
                    'scope_of_work' => $tender->scope_of_work ?? '',
                    'line_items' => $tender->lineItems ?? collect([]),
                    'subject' => $subject,
                    'customer_id' => $ids['customer_id'],
                    'workspace_id' => $ids['workspace_id'],
                ])->render();
                // Send email using EmailTrait
                $params = [
                    'to' => $subcontractor->email,
                    'subject' => $subject,
                    'msg' => $emailContent,
                ];
                if ($this->SendInstantEmail($params)) {
                    $emailSentCount++;
                } else {
                    $errors[] = "Failed to send email to: {$subcontractor->email}";
                }
            } 
        $response = [
            'invited_count' => $invitedCount,
            'email_sent_count' => $emailSentCount,
            'total_requested' => count($subcontractorIds),
        ];
        if (!empty($errors)) {
            $response['errors'] = $errors;
        }
        $message = "Successfully invited {$invitedCount} subcontractor(s). {$emailSentCount} email(s) sent.";
        if (!empty($errors)) {
            $message .= " Some errors occurred.";
        }
        return $this->success($response, $message);
    }

    /**
     * Publish tender
     * Updates is_published to 1 and changes status from draft to open
     */
    public function publish(Request $request)
    {
        $query = Tender::query();
        $query = $this->applyCustomerWorkspaceFilter($query);
        $tender = $query->find($request->tender_id);
        if (!$tender) {
            return $this->message('Tender not found or you do not have access', 404);
        }
        $updateData = [
            'is_published' => 1,
        ];
        if ($tender->status == 'draft') {
            $updateData['status'] = 'open';
        }
        $tender->update($updateData);
        return $this->success($tender, 'Tender published successfully');
    }

   

    /**
     * Upload document to tender (Admin only)
     */
    public function uploadTenderDocument(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'tender_id' => 'required|integer|exists:tenders,id',
            'title' => 'required|string|max:255',
            'file' => 'required|file|mimes:pdf,doc,docx,xlsx,jpg,jpeg,png|max:10240',
        ]);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        $ids = $this->getCustomerAndWorkspaceIds();
        if (!$ids) {
            return $this->message('Unable to determine customer or workspace', 400);
        }

        $query = Tender::query();
        $query = $this->applyCustomerWorkspaceFilter($query);
        $tender = $query->find($request->tender_id);

        if (!$tender) {
            return $this->message('Tender not found or you do not have access', 404);
        }
        $uploadPath = 'tender_documents';
        $fullUploadPath = public_path($uploadPath);
        if (!file_exists($fullUploadPath)) {
            mkdir($fullUploadPath, 0755, true);
        }
        $file = $request->file('file');
        $filename = time() . '_' . uniqid() . '_' . $file->getClientOriginalName();
        $file->move($fullUploadPath, $filename);
        $filePath = $uploadPath . '/' . $filename;
        $attachment = TenderAttachment::create([
            'entity_type' => 'customer',
            'entity_id' => $tender->id,
            'title' => $request->title,
            'file_name' => $filename,
            'file_path' => $filePath,
            'uploaded_by' => Auth::id(),
            'customer_id' => $ids['customer_id'],
            'workspace_id' => $ids['workspace_id'],
        ]);
        return $this->success($attachment, 'Document uploaded successfully');
    }

    /**
     * Get all documents for a tender (Admin)
     */
    public function getTenderDocuments($tenderId)
    {
        $query = Tender::query();
        $query = $this->applyCustomerWorkspaceFilter($query);
        $tender = $query->find($tenderId);
        if (!$tender) {
            return $this->message('Tender not found or you do not have access', 404);
        }
        $documents = TenderAttachment::where('entity_type', 'customer')
            ->where('entity_id', $tenderId)
            ->with('uploadedBy')
            ->orderBy('created_at', 'desc')
            ->get();

        return $this->success($documents, 'Tender documents retrieved successfully');
    }

    /**
     * Delete tender document (Admin)
     */
    public function deleteTenderDocument(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'document_id' => 'required|integer|exists:tender_attachments,id',
            'tender_id' => 'required|integer|exists:tenders,id',
        ]);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        $query = Tender::query();
        $query = $this->applyCustomerWorkspaceFilter($query);
        $tender = $query->find($request->tender_id);
        if (!$tender) {
            return $this->message('Tender not found or you do not have access', 404);
        }
        $document = TenderAttachment::where('id', $request->document_id)
        ->where('entity_type', 'customer')
        ->where('entity_id', $tender->id)
        ->first();
        if (!$document) {
            return $this->message('Document not found or you do not have access', 404);
        }
        // Delete file from storage
        if ($document->file_path && file_exists(public_path($document->file_path))) {
            unlink(public_path($document->file_path));
        }
        $document->delete();
        return $this->message('Document deleted successfully', 200);
    }

    /**
     * Get all bids for a tender (Admin)
     */
    public function getAllBidsForTender($tenderId)
    {
        $query = Tender::query();
        $query = $this->applyCustomerWorkspaceFilter($query);
        $tender = $query->find($tenderId);
        if (!$tender) {
            return $this->message('Tender not found or you do not have access', 404);
        }
        // Get all unique users who submitted bids
        $userIds = TenderLineItemsBid::where('tender_id', $tenderId)
            ->where('status', 'submitted')
            ->distinct()
            ->pluck('user_id');
        $bids = [];
        foreach ($userIds as $userId) {
            $user = User::find($userId);
            $lineItemBids = TenderLineItemsBid::where('tender_id', $tenderId)
                ->where('user_id', $userId)
                ->where('status', 'submitted')
                ->with('lineItem')
                ->get();
            $firstBid = $lineItemBids->first();
            $bids[] = [
                'user_id' => $userId,
                'user_name' => $user->name ?? 'N/A',
                'user_email' => $user->email ?? 'N/A',
                'total_base_bid' => $firstBid->total_base_bid ?? 0,
                'exclusions_clarifications' => $firstBid->exclusions_clarifications ?? null,
                'is_late_submission' => $firstBid->is_late_submission ?? false,
                'submitted_at' => $firstBid->submitted_at ?? null,
                'line_items' => $lineItemBids->map(function ($bid) {
                    return [
                        'line_item_id' => $bid->tender_line_items_id,
                        'description' => $bid->lineItem->description ?? 'N/A',
                        'unit' => $bid->lineItem->unit ?? 'N/A',
                        'quantity' => $bid->lineItem->quantity ?? 0,
                        'bid_price' => $bid->bid_price,
                    ];
                }),
                'attachments' => TenderAttachment::where('entity_type', 'tender_bid')
                    ->where('entity_id', $tenderId . '_' . $userId)
                    ->get(),
            ];
        }
        return $this->success($bids, 'All bids retrieved successfully');
    }

    /**
     * Compare bids for a tender (Admin)
     */
    public function compareBids(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'tender_id' => 'required|integer|exists:tenders,id',
            'user_ids' => 'required|array|min:2',
            'user_ids.*' => 'required|integer|exists:users,id',
        ]);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        $query = Tender::query();
        $query = $this->applyCustomerWorkspaceFilter($query);
        $tender = $query->find($request->tender_id);
        if (!$tender) {
            return $this->message('Tender not found or you do not have access', 404);
        }
        $tender->load('lineItems');
        $comparison = [
            'tender_id' => $tender->id,
            'tender_title' => $tender->title,
            'tender_reference' => $tender->reference,
            'line_items' => [],
            'bids_summary' => [],
        ];
        // Get line items
        foreach ($tender->lineItems as $lineItem) {
            $lineItemData = [
                'line_item_id' => $lineItem->id,
                'description' => $lineItem->description,
                'unit' => $lineItem->unit,
                'quantity' => $lineItem->quantity,
                'bids' => [],
            ];
            // Get bids for each user
            foreach ($request->user_ids as $userId) {
                $bid = TenderLineItemsBid::where('tender_id', $tender->id)
                    ->where('user_id', $userId)
                    ->where('tender_line_items_id', $lineItem->id)
                    ->where('status', 'submitted')
                    ->first();

                $user = User::find($userId);
                $lineItemData['bids'][] = [
                    'user_id' => $userId,
                    'user_name' => $user->name ?? 'N/A',
                    'bid_price' => $bid->bid_price ?? 0,
                ];
            }
            $comparison['line_items'][] = $lineItemData;
        }
        // Get summary for each bid
        foreach ($request->user_ids as $userId) {
            $firstBid = TenderLineItemsBid::where('tender_id', $tender->id)
                ->where('user_id', $userId)
                ->where('status', 'submitted')
                ->first();
            $user = User::find($userId);
            $comparison['bids_summary'][] = [
                'user_id' => $userId,
                'user_name' => $user->name ?? 'N/A',
                'user_email' => $user->email ?? 'N/A',
                'total_base_bid' => $firstBid->total_base_bid ?? 0,
                'is_late_submission' => $firstBid->is_late_submission ?? false,
                'submitted_at' => $firstBid->submitted_at ?? null,
            ];
        }
        return $this->success($comparison, 'Bid comparison retrieved successfully');
    }


     /**
     * Get RFIs for a tender (Admin - shows all RFIs)
     */
    public function getRfisForTender($tenderId)
    {
        $tender = Tender::find($tenderId);
        if (!$tender) {
            return $this->message('Tender not found', 404);
        }
        $query = Tender::query();
        $query = $this->applyCustomerWorkspaceFilter($query);
        $tender = $query->find($tenderId);
        if (!$tender) {
            return $this->message('Tender not found or you do not have access', 404);
        }
        // Admin sees all RFIs
        $rfis = TenderRfi::with('user')
            ->where('tender_id', $tenderId)
            ->orderBy('created_at', 'desc')
            ->get();
        return $this->success($rfis, 'RFIs retrieved successfully');
    }


    /**
     * Reply to RFI question (Admin)
     */
    public function replyToRfi(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'rfi_id' => 'required|integer|exists:tender_rfis,id',
            'answer' => 'required|string',
        ]);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        $ids = $this->getCustomerAndWorkspaceIds();
        $rfi = TenderRfi::where('id', $request->rfi_id)
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->first();
        if (!$rfi) {
            return $this->message('RFI not found or you do not have access', 404);
        }
        $rfi->update([
            'answer' => $request->answer,
            'status' => 'answered',
            'answered_at' => now(),
        ]);
        return $this->success($rfi, 'RFI answered successfully');
    }

    /**
     * Assign tender to subcontractor (Admin)
     */
    public function assignTender(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'tender_id' => 'required|integer|exists:tenders,id',
            'user_id' => 'required|integer|exists:users,id',
        ]);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        $ids = $this->getCustomerAndWorkspaceIds();
        if (!$ids) {
            return $this->message('Unable to determine customer or workspace', 400);
        }
        $query = Tender::query();
        $query = $this->applyCustomerWorkspaceFilter($query);
        $tender = $query->find($request->tender_id);
        if (!$tender) {
            return $this->message('Tender not found or you do not have access', 404);
        }
        $participant = TenderParticipant::where('tender_id', $tender->id)
            ->where('user_id', $request->user_id)
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->first();
        if (!$participant) {
            return $this->message('Subcontractor is not a participant in this tender', 404);
        }
        // Unassign all other participants
        TenderParticipant::where('tender_id', $tender->id)
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->update(['is_assigned' => false, 'assigned_at' => null]);
        // Assign to selected participant
        $participant->update([
            'is_assigned' => true,
            'assigned_at' => now(),
        ]);
        return $this->success($participant, 'Tender assigned successfully');
    }
}
