<?php

namespace App\Http\Controllers\Traits;

use App\Models\SiteDiary;
use App\Models\SiteDiaryEquipment;
use App\Models\SiteDiaryTask;
use App\Models\SiteDiaryImage;
use App\Models\SiteDiaryLog;
use App\Models\EmpCompanyDetails;
use App\Models\EmpPersonalDetails;
use App\Models\Sites;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;

trait SiteDiaryTrait
{
    /**
     * Log changes in site diary system
     *
     * @param string $action_type create|edit|delete|complete|incomplete
     * @param mixed $loggable The model instance being logged (SiteDiary|SiteDiaryEquipment|SiteDiaryTask|SiteDiaryImage)
     * @param array $oldValues Previous values before change
     * @param array $newValues New values after change
     * @param string|null $notes Additional notes about the change
     * @return void
     */
    protected function logSiteDiaryAction($action_type, $loggable, $oldValues = null, $newValues = null, $notes = null)
    {
        try {
            $ids = $this->getCustomerAndWorkspaceIds();
            // Get the site diary ID
            $siteDiaryId = $loggable instanceof SiteDiary
                ? $loggable->id
                : $loggable->site_diary_id;
            // Determine if action is by employee or customer
            $userId = null;
            $empId = null;
            if ($ids['flag'] === 'emp') {
                $empId = Auth::id();
                $createdByType = 'employee';
            } else {
                $userId = Auth::id();
                $createdByType = 'customer';
            }
            // Create the log entry
            SiteDiaryLog::create([
                'site_diary_id' => $siteDiaryId,
                'loggable_type' => get_class($loggable),
                'loggable_id' => $loggable->id,
                'user_id' => $userId,
                'emp_id' => $empId,
                'action_type' => $action_type,
                'old_values' => $oldValues,
                'new_values' => $newValues,
                'notes' => $notes,
                'created_by_type' => $createdByType,
                'customer_id' => $ids['customer_id'],
                'workspace_id' => $ids['workspace_id'],
            ]);
        } catch (\Exception $e) {
            Log::error('Failed to create site diary log: ' . $e->getMessage());
        }
    }


    protected function getSiteDiaries(Request $request)
    {
        $ids = $this->getCustomerAndWorkspaceIds();        
        $query = SiteDiary::where('del', '0');
        $query = $this->applyCustomerWorkspaceFilter($query);
        $date = $request->filled('diary_date')
            ? Carbon::parse($request->input('diary_date'))->toDateString()
            : now()->toDateString();
        $query->whereDate('diary_date', $date);
        if($request->filled('site_id')){
            $query->where('site_id', intval($request->site_id));
        }
        $siteDiary = $query->with([
            'site',
            'equipment',
            'tasks',
            'images',
        ])->first();

        if (!$siteDiary) {
            return $this->error('No site diary found for the specified criteria', 404);
        }

        // Get tier permissions and site permissions for the logged-in user
        $tierPermissions = [];
        
        if ($ids['flag'] === 'emp') {
            $employeeId = Auth::id();
            $employee = EmpCompanyDetails::find($employeeId);
            
            if ($employee) {
                // Get only "Sites" module permissions using the global function
                $sitesModulePermission = $this->getTierPermissionsByModule($employee, 'Sites');
                if ($sitesModulePermission) {
                    $tierPermissions = [$sitesModulePermission];
                }
            }
        }

        // Transform the data to group logs by type
        $groupedLogs = [
            'diary_logs' => [],
            'equipment_logs' => [],
            'task_logs' => [],
            'image_logs' => []
        ];

        foreach ($siteDiary->allLogs as $log) {
            $logData = [
                'id' => $log->id,
                'site_diary_id' => $log->site_diary_id,
                'loggable_type' => $log->loggable_type,
                'loggable_id' => $log->loggable_id,
                'user_id' => $log->user_id,
                'emp_id' => $log->emp_id,
                'action_type' => $log->action_type,
                'old_values' => $log->old_values,
                'new_values' => $log->new_values,
                'notes' => $log->notes,
                'created_at' => $log->created_at,
                'updated_at' => $log->updated_at,
                'created_by_type' => $log->created_by_type,
                'creator_info' => $log->creator_info
            ];

            // Group logs by model type
            switch ($log->loggable_type) {
                case 'App\\Models\\SiteDiary':
                    $groupedLogs['diary_logs'][] = $logData;
                    break;
                case 'App\\Models\\SiteDiaryEquipment':
                    $groupedLogs['equipment_logs'][] = $logData;
                    break;
                case 'App\\Models\\SiteDiaryTask':
                    $groupedLogs['task_logs'][] = $logData;
                    break;
                case 'App\\Models\\SiteDiaryImage':
                    $groupedLogs['image_logs'][] = $logData;
                    break;
            }
        }

        // Replace all_logs with grouped logs
        $siteDiary->setRelation('allLogs', collect($groupedLogs));
        // Add tier permissions and site permissions to response
        $responseData = $siteDiary->toArray();
        // Add Sites module permissions (array format)
        $responseData['tier_permissions'] = $tierPermissions;
        return $this->success($responseData, 'Site diaries retrieved successfully');
    }

    protected function createSiteDiary(Request $request)
    {

        $validator = $this->siteDiaryValidation($request);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        $ids = $this->getCustomerAndWorkspaceIds();
        $data = $validator->validated();
        // Check if site diary already exists for this site and date
        $existingDiary = SiteDiary::where('site_id', $data['site_id'])
            ->whereDate('diary_date', $data['diary_date'] ?? now()->format('Y-m-d'))
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->where('del', '0')
            ->first();
        if ($existingDiary) {
            return $this->error('Site diary already exists for this site on ' . $existingDiary->diary_date . ' this date', 400);
        }
        try {
            DB::beginTransaction();
            // Prepare site diary data
            $siteDiaryData = [
                'site_id' => $data['site_id'],
                'diary_date' => $data['diary_date'] ?? now()->format('Y-m-d'), // Use current date if not provided
                'weather' => $data['weather'] ?? null,
                'work_log' => $data['work_log'] ?? null,
                'notes' => $data['notes'] ?? null,
                'customer_id' => $ids['customer_id'],
                'workspace_id' => $ids['workspace_id'],
                'summary' => $data['summary'] ?? null,
                'created_by_type' => $ids['flag'] === 'emp' ? 'employee' : 'customer',
                'created_by' => Auth::id()
            ];
            $siteDiary = SiteDiary::create($siteDiaryData);            
            // Log site diary creation
            $this->logSiteDiaryAction('create', $siteDiary, null, $siteDiaryData, 'Site diary created');

            // Handle equipment
            if (!empty($data['equipment'])) {
                foreach ($data['equipment'] as $equipment) {
                    $equipmentModel = SiteDiaryEquipment::create([
                        'site_diary_id' => $siteDiary->id,
                        'equipment_name' => $equipment['equipment_name'],
                        'quantity' => $equipment['quantity'],
                        'customer_id' => $ids['customer_id'],
                        'workspace_id' => $ids['workspace_id'],
                    ]);

                    $this->logSiteDiaryAction('create', $equipmentModel, null, $equipment, 'Equipment added to site diary');
                }
            }
            // Handle tasks
            if (!empty($data['tasks'])) {
                foreach ($data['tasks'] as $task) {
                    $taskModel = SiteDiaryTask::create([
                        'site_diary_id' => $siteDiary->id,
                        'title' => $task['title'],
                        'is_completed' => $task['is_completed'] ?? false,
                        'customer_id' => $ids['customer_id'],
                        'workspace_id' => $ids['workspace_id'],
                    ]);

                    // Log task creation
                    $this->logSiteDiaryAction('create', $taskModel, null, $task, 'Task added to site diary');
                }
            }
            // Handle images
            if ($request->hasFile('images')) {
                foreach ($request->file('images') as $imageFile) {
                    if ($imageFile->isValid()) {
                        // Generate unique filename
                        $extension = strtolower($imageFile->getClientOriginalExtension());
                        $originalName = pathinfo($imageFile->getClientOriginalName(), PATHINFO_FILENAME);
                        $sanitizedFileName = preg_replace('/[^a-zA-Z0-9_-]/', '', $originalName);
                        $sanitizedFileName = substr($sanitizedFileName, 0, 100);
                        $uniqueFileName = time() . '_' . uniqid() . '_' . $sanitizedFileName . '.' . $extension;
                        // Create directory if it doesn't exist
                        $directory = public_path('SiteDiaries');
                        if (!file_exists($directory)) {
                            mkdir($directory, 0777, true);
                        }
                        // Move uploaded file
                        $imageFile->move($directory, $uniqueFileName);
                        $imagePath = 'SiteDiaries/' . $uniqueFileName;
                        // Save to database
                        $imageModel = SiteDiaryImage::create([
                            'site_diary_id' => $siteDiary->id,
                            'image' => $imagePath,
                            'customer_id' => $ids['customer_id'],
                            'workspace_id' => $ids['workspace_id'],
                        ]);

                        // Log image creation
                        $this->logSiteDiaryAction('create', $imageModel, null, ['image' => $imagePath], 'Image added to site diary');
                    }
                }
            }
            DB::commit();
            return $this->message('Site diary created successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to create site diary: ' . $e->getMessage());
            return $this->error('Failed to create site diary' . $e->getMessage(), 500);
        }
    }

    protected function getSiteDiaryById($id, $ids)
    {
        $siteDiary = SiteDiary::with([
            'site',
            'equipment',
            'tasks',
            'images',
            'createdByEmployee',
            'createdByCustomer',
        ])
            ->where('id', $id)
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->where('del', '0')
            ->first();
        if (!$siteDiary) {
            return $this->error('Site diary not found', 404);
        }
        return $this->success($siteDiary, 'Site diary retrieved successfully');
    }

    protected function updateSiteDiary(Request $request)
    {
        $validator = $this->siteDiaryValidation($request, true);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        $ids = $this->getCustomerAndWorkspaceIds();
        $data = $validator->validated();
        $siteDiary = SiteDiary::where('id', $data['id'])
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->where('del', '0')
            ->first();
        if (!$siteDiary) {
            return $this->error('Site diary not found', 404);
        }
        // Check if another site diary exists for this site and date (excluding current one)
        $existingDiary = SiteDiary::where('site_id', $data['site_id'])
            ->whereDate('diary_date', $data['diary_date'] ?? now()->format('Y-m-d'))
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->where('del', '0')
            ->where('id', '!=', $data['id'])
            ->first();
        if ($existingDiary) {
            return $this->error('Site diary already exists for this site on this date', 400);
        }
        try {
            DB::beginTransaction();
            $oldValues = $siteDiary->toArray();
            // Update site diary
            $newValues = [
                'site_id' => $data['site_id'],
                'diary_date' => $data['diary_date'] ?? now()->format('Y-m-d'),
                'weather' => $data['weather'] ?? null,
                'work_log' => $data['work_log'] ?? null,
                'notes' => $data['notes'] ?? null,
                'summary' => $data['summary'] ?? null,
            ];
            $siteDiary->update($newValues);
            // Log the update
            $this->logSiteDiaryAction('update', $siteDiary, $oldValues, $newValues, 'Site diary updated');
            // Update equipment (delete existing and create new)
            $existingEquipment = $siteDiary->equipment()->get();
            
            // Log deletion of existing equipment
            foreach ($existingEquipment as $equipment) {
                $oldValues = $equipment->toArray();
                // $this->logSiteDiaryAction('delete', $equipment, $oldValues, null, 'Equipment removed from site diary');
            }
            $siteDiary->equipment()->delete();          
            if (!empty($data['equipment'])) {
                foreach ($data['equipment'] as $equipment) {
                    $equipmentModel = SiteDiaryEquipment::create([
                        'site_diary_id' => $siteDiary->id,
                        'equipment_name' => $equipment['equipment_name'],
                        'quantity' => $equipment['quantity'],
                        'customer_id' => $ids['customer_id'],
                        'workspace_id' => $ids['workspace_id'],
                    ]);
                    // Log equipment as create with complete new values
                    $newValues = $equipmentModel->toArray();
                    $this->logSiteDiaryAction('update', $equipmentModel, null, $newValues, 'Equipment added to site diary');
                }
            }
            // Update tasks (delete existing and create new)
            $existingTasks = $siteDiary->tasks()->get();
            
            // Log deletion of existing tasks
            foreach ($existingTasks as $task) {
                $oldValues = $task->toArray();
                // $this->logSiteDiaryAction('delete', $task, $oldValues, null, 'Task removed from site diary');
            }
            
            $siteDiary->tasks()->delete();
            if (!empty($data['tasks'])) {
                foreach ($data['tasks'] as $task) {
                    $taskModel = SiteDiaryTask::create([
                        'site_diary_id' => $siteDiary->id,
                        'title' => $task['title'],
                        'is_completed' => $task['is_completed'] ?? false,
                        'customer_id' => $ids['customer_id'],
                        'workspace_id' => $ids['workspace_id'],
                    ]);
                    // Log task as create with complete new values
                    $newValues = $taskModel->toArray();
                    $this->logSiteDiaryAction('update', $taskModel, null, $newValues, 'Task added to site diary');
                }
            }

            $existingImages = $siteDiary->images()->get();
            // Log deletion of existing images
            foreach ($existingImages as $existingImage) {
                $oldImageValues = $existingImage->toArray();
                // $this->logSiteDiaryAction('delete', $existingImage, $oldImageValues, null, 'Site diary image marked as deleted');
            }
            // Delete existing images
            $siteDiary->images()->delete();
            
            // Handle new images
            if ($request->hasFile('images')) {
                $uploadedBy = Auth::id();
                foreach ($request->file('images') as $imageFile) {
                    if ($imageFile->isValid()) {
                        // Generate unique filename
                        $extension = strtolower($imageFile->getClientOriginalExtension());
                        $originalName = pathinfo($imageFile->getClientOriginalName(), PATHINFO_FILENAME);
                        $sanitizedFileName = preg_replace('/[^a-zA-Z0-9_-]/', '', $originalName);
                        $sanitizedFileName = substr($sanitizedFileName, 0, 100);
                        $uniqueFileName = time() . '_' . uniqid() . '_' . $sanitizedFileName . '.' . $extension;
                        // Create directory if it doesn't exist
                        $directory = public_path('SiteDiaries');
                        if (!file_exists($directory)) {
                            mkdir($directory, 0777, true);
                        }
                        // Move uploaded file
                        $imageFile->move($directory, $uniqueFileName);
                        $imagePath = 'SiteDiaries/' . $uniqueFileName;
                        // Save to database
                        $imageModel = SiteDiaryImage::create([
                            'site_diary_id' => $siteDiary->id,
                            'image' => $imagePath,
                            'uploaded_by' => $uploadedBy,
                            'customer_id' => $ids['customer_id'],
                            'workspace_id' => $ids['workspace_id'],
                        ]);
                        // Log image as create with complete new values
                        $newValues = $imageModel->toArray();
                        $this->logSiteDiaryAction('update', $imageModel, null, $newValues, 'Image added to site diary');
                    }
                }
            }
            DB::commit();
            // Load relationships for response
            return $this->message('Site diary updated successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to update site diary: ' . $e->getMessage());
            return $this->error('Failed to update site diary' . $e->getMessage(), 500);
        }
    }

    protected function deleteSiteDiary($id, $ids)
    {
        $siteDiary = SiteDiary::where('id', $id)
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->where('del', '0')
            ->first();

        if (!$siteDiary) {
            return $this->error('Site diary not found', 404);
        }

        $oldValues = $siteDiary->toArray();
        $siteDiary->del = 1;
        $siteDiary->save();

        // Log the deletion
        $this->logSiteDiaryAction('delete', $siteDiary, $oldValues, $siteDiary->toArray(), 'Site diary marked as deleted');

        return $this->message('Site diary deleted successfully');
    }

    protected function deleteSiteDiaryImage($id, $ids)
    {
        $image = SiteDiaryImage::where('id', $id)
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            ->first();
        if (!$image) {
            return $this->error('Site diary image not found', 404);
        }
        $oldValues = $image->toArray();
        $image->delete();
        $this->logSiteDiaryAction('delete', $image, $oldValues, null, 'Site diary image marked as deleted');
        return $this->message('Site diary image deleted successfully');
    }

    protected function toggleTaskStatus(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'task_id' => 'required|integer|exists:site_diary_tasks,id',
            'is_completed' => 'required|boolean',
        ]);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        $ids = $this->getCustomerAndWorkspaceIds();
        $validatedData = $validator->validated();
        $task = SiteDiaryTask::where('id', $validatedData['task_id'])
            ->where('customer_id', $ids['customer_id'])
            ->where('workspace_id', $ids['workspace_id'])
            // ->withTrashed()  
            ->first();
        if (!$task) {
            return $this->error('Task not found', 404);
        }
        $oldValues = $task->toArray();
        $task->is_completed = $request->is_completed;
        $task->save();
        // Log the task status change
        $actionType = $request->is_completed ? 'complete' : 'incomplete';
        $this->logSiteDiaryAction(
            $actionType,
            $task,
            $oldValues,
            $task->toArray(),
            'Task status changed to ' . ($request->is_completed ? 'completed' : 'incomplete')
        );
        return $this->success($task, 'Task status updated successfully');
    }

    /**
     * Generate AI summary for site diary
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    protected function generateAiSummary(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'site_id' => 'required|integer',
            'diary_date' => 'nullable|date',
            'site_diary_id' => 'required|integer|exists:site_diaries,id',
        ]);

        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }

        $ids = $this->getCustomerAndWorkspaceIds();
        $data = $validator->validated();

        try {
            // Find the site diary
            $query = SiteDiary::where('del', '0')
                ->where('customer_id', $ids['customer_id'])
                ->where('workspace_id', $ids['workspace_id'])
                ->where('site_id', $data['site_id']);

            // Add date filter if provided, otherwise use current date
            if (!empty($data['diary_date'])) {
                $query->whereDate('diary_date', $data['diary_date']);
            } else {
                $query->whereDate('diary_date', now()->format('Y-m-d'));
            }

            if (isset($data['site_diary_id'])) {
                $query->where('id', $data['site_diary_id']);
            }

            $siteDiary = $query->with([
                'site',
                'equipment',
                'tasks',
                'images',
                'allLogs.createByUser',
                'allLogs.createByEmployee'
            ])->first();

            if (!$siteDiary) {
                return $this->error('No site diary found for the specified criteria', 404);
            }


            // Collect all data for AI summary
            $summaryData = $this->collectSiteDiaryData($siteDiary);

            // Generate AI summary using ChatGPT
            $aiSummary = $this->generateChatGptSummary($summaryData);

            if (!$aiSummary) {
                return $this->error('Failed to generate AI summary', 500);
            }

            // Update the site diary with the generated summary
            $oldValues = $siteDiary->toArray();
            $siteDiary->summary = $aiSummary;
            $siteDiary->save();

            // Log the summary generation
            $this->logSiteDiaryAction(
                'edit',
                $siteDiary,
                $oldValues,
                $siteDiary->toArray(),
                'AI summary generated and saved'
            );

            return $this->success([
                'site_diary_id' => $siteDiary->id,
                'summary' => $aiSummary,
                'generated_at' => now()->toDateTimeString()
            ], 'AI summary generated successfully');

        } catch (\Exception $e) {
            Log::error('Failed to generate AI summary: ' . $e->getMessage());
            return $this->error('Failed to generate AI summary: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Collect all site diary data for AI processing
     *
     * @param SiteDiary $siteDiary
     * @return array
     */
    private function collectSiteDiaryData($siteDiary)
    {
        $data = [
            'basic_info' => [
                'site_name' => $siteDiary->site->title ?? 'N/A',
                'site_description' => $siteDiary->site->description ?? 'N/A',
                'diary_date' => $siteDiary->diary_date,
                'weather' => $siteDiary->weather ?? 'N/A',
            ],
            'work_log' => $siteDiary->work_log ?? 'No work log provided',
            'notes' => $siteDiary->notes ?? 'No notes provided',
            'equipment' => [],
            'tasks' => [],
            'activity_logs' => [],
        ];

        // Collect equipment data
        foreach ($siteDiary->equipment as $equipment) {
            $data['equipment'][] = [
                'name' => $equipment->equipment_name,
                'quantity' => $equipment->quantity,
            ];
        }

        // Collect tasks data
        foreach ($siteDiary->tasks as $task) {
            $data['tasks'][] = [
                'title' => $task->title,
                'status' => $task->is_completed ? 'Completed' : 'Pending',
            ];
        }

        // Collect activity logs
        foreach ($siteDiary->allLogs as $log) {
            $creatorInfo = $log->createByUser ?? $log->createByEmployee;
            $creatorName = 'Unknown';
            
            if ($creatorInfo) {
                if ($log->created_by_type === 'employee' && $log->createByEmployee) {
                    $creatorName = trim(($log->createByEmployee->first_name ?? '') . ' ' . ($log->createByEmployee->last_name ?? ''));
                } elseif ($log->created_by_type === 'customer' && $log->createByUser) {
                    $creatorName = $log->createByUser->name ?? 'Unknown';
                }
            }

            $data['activity_logs'][] = [
                'action' => $log->action_type,
                'type' => $log->loggable_type,
                'notes' => $log->notes,
                'created_by' => $creatorName,
                'created_at' => $log->created_at ? Carbon::parse($log->created_at)->format('Y-m-d H:i:s') : 'Unknown',
            ];
        }

        return $data;
    }

    /**
     * Generate summary using ChatGPT API
     *
     * @param array $data
     * @return string|null
     */
    private function generateChatGptSummary($data)
    {
        try {
            $apiKey = config('constants.chat_gpt_api_key');
            
            if (!$apiKey) {
                throw new \Exception('ChatGPT API key not configured');
            }

            // Prepare the prompt
            $prompt = $this->buildSummaryPrompt($data);

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $apiKey,
                'Content-Type' => 'application/json',
            ])->timeout(60)->post('https://api.openai.com/v1/chat/completions', [
                'model' => 'gpt-3.5-turbo',
                'messages' => [
                    [
                        'role' => 'system',
                        'content' => 'You are an expert construction site diary analyst. Generate concise, professional summaries of site diary activities. CRITICAL REQUIREMENT: Your summary must be exactly 150 words and in australian english - no more, no less. Count the words carefully and ensure you meet this exact requirement. Focus on key accomplishments, equipment usage, task progress, weather impact, and important activities in exactly 150 words.'
                    ],
                    [
                        'role' => 'user',
                        'content' => $prompt
                    ]
                ],
                'temperature' => 0.7,
                'max_tokens' => 250
            ]);

            if (!$response->successful()) {
                Log::error('ChatGPT API error', [
                    'status' => $response->status(),
                    'response' => $response->body()
                ]);
                return null;
            }

            $aiResponse = $response->json();
            $summary = $aiResponse['choices'][0]['message']['content'] ?? null;
            
            if ($summary) {
                // Validate word count and trim if necessary
                $summary = $this->ensureExact150Words($summary);
            }
            
            return $summary;

        } catch (\Exception $e) {
            Log::error('Failed to generate ChatGPT summary: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Build prompt for AI summary generation
     *
     * @param array $data
     * @return string
     */
    private function buildSummaryPrompt($data)
    {
        $prompt = "Please analyze the following site diary data and generate a comprehensive summary:\n\n";
        
        // Basic information
        $prompt .= "**Site Information:**\n";
        $prompt .= "- Site: {$data['basic_info']['site_name']}\n";
        $prompt .= "- Date: {$data['basic_info']['diary_date']}\n";
        $prompt .= "- Weather: {$data['basic_info']['weather']}\n";
        if (!empty($data['basic_info']['site_description'])) {
            $prompt .= "- Description: {$data['basic_info']['site_description']}\n";
        }
        $prompt .= "\n";

        // Work log
        if (!empty($data['work_log']) && $data['work_log'] !== 'No work log provided') {
            $prompt .= "**Work Log:**\n{$data['work_log']}\n\n";
        }

        // Notes
        if (!empty($data['notes']) && $data['notes'] !== 'No notes provided') {
            $prompt .= "**Notes:**\n{$data['notes']}\n\n";
        }

        // Equipment
        if (!empty($data['equipment'])) {
            $prompt .= "**Equipment Used:**\n";
            foreach ($data['equipment'] as $equipment) {
                $prompt .= "- {$equipment['name']} (Qty: {$equipment['quantity']})\n";
            }
            $prompt .= "\n";
        }

        // Tasks
        if (!empty($data['tasks'])) {
            $prompt .= "**Tasks:**\n";
            foreach ($data['tasks'] as $task) {
                $prompt .= "- {$task['title']} - Status: {$task['status']}\n";
            }
            $prompt .= "\n";
        }

        // Activity logs (last 10 most recent)
        if (!empty($data['activity_logs'])) {
            $prompt .= "**Recent Activities:**\n";
            $recentLogs = array_slice($data['activity_logs'], -10); // Get last 10 activities
            foreach ($recentLogs as $log) {
                $prompt .= "- {$log['action']} by {$log['created_by']} at {$log['created_at']}";
                if (!empty($log['notes'])) {
                    $prompt .= " - {$log['notes']}";
                }
                $prompt .= "\n";
            }
        }

        $prompt .= "\nPlease provide a professional summary that includes:\n";
        $prompt .= "1. Overall progress and achievements\n";
        $prompt .= "2. Key equipment utilization\n";
        $prompt .= "3. Task completion status\n";
        $prompt .= "4. Weather impact (if any)\n";
        $prompt .= "5. Notable activities or concerns\n";
        $prompt .= "6. Next steps or recommendations (if applicable)\n\n";
        $prompt .= "IMPORTANT: Your response must be exactly 150 words. Count carefully and ensure you do not exceed or fall short of this requirement.";

        return $prompt;
    }

    /**
     * Ensure the summary is exactly 150 words
     *
     * @param string $summary
     * @return string
     */
    private function ensureExact150Words($summary)
    {
        // Clean and split into words
        $summary = trim($summary);
        $words = preg_split('/\s+/', $summary, -1, PREG_SPLIT_NO_EMPTY);
        $wordCount = count($words);

        if ($wordCount === 150) {
            return $summary;
        }

        if ($wordCount > 150) {
            // Trim to exactly 150 words
            $trimmedWords = array_slice($words, 0, 150);
            $trimmedSummary = implode(' ', $trimmedWords);
            
            // Ensure it ends with proper punctuation
            if (!in_array(substr($trimmedSummary, -1), ['.', '!', '?'])) {
                $trimmedSummary .= '.';
            }
            
            Log::info("AI Summary trimmed from {$wordCount} to 150 words");
            return $trimmedSummary;
        }

        if ($wordCount < 150) {
            // If less than 150 words, add a note about the shortage
            $wordsNeeded = 150 - $wordCount;
            Log::warning("AI Summary only {$wordCount} words, needed 150 (short by {$wordsNeeded} words)");
            
            // Add a brief note to reach closer to 150 words
            $summary .= " Additional monitoring and documentation will continue to ensure comprehensive project tracking and safety compliance throughout the construction process.";
            
            // Re-check and trim if we went over
            $words = preg_split('/\s+/', $summary, -1, PREG_SPLIT_NO_EMPTY);
            if (count($words) > 150) {
                $trimmedWords = array_slice($words, 0, 150);
                $summary = implode(' ', $trimmedWords);
                if (!in_array(substr($summary, -1), ['.', '!', '?'])) {
                    $summary .= '.';
                }
            }
        }

        return $summary;
    }
}
