<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Message extends BaseModel
{
    use HasFactory;

    protected $fillable = [
        'sender_id',
        'sender_type',
        'receiver_id',
        'receiver_type',
        'message',
        'conversation_id',
        'is_read',
        'message_type',
        'attachment_path',
        'customer_id',
        'workspace_id'
    ];

    protected $hidden = [
        'customer_id',
        'workspace_id',
    ];
    
    protected $casts = [
        'is_read' => 'boolean',
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    public function sender()
    {
        if ($this->sender_type === 'emp') {
            return $this->belongsTo(EmpCompanyDetails::class, 'sender_id');
        }
        return $this->belongsTo(User::class, 'sender_id');
    }

    public function receiver()
    {
        if ($this->receiver_type === 'emp') {
            return $this->belongsTo(EmpCompanyDetails::class, 'receiver_id');
        }
        return $this->belongsTo(User::class, 'receiver_id');
    }

    // Separate relationships for easier loading
    public function senderEmployee()
    {
        return $this->belongsTo(EmpCompanyDetails::class, 'sender_id');
    }

    public function senderUser()
    {
        return $this->belongsTo(User::class, 'sender_id');
    }

    public function conversation()
    {
        return $this->belongsTo(Conversation::class);
    }

    public function customer()
    {
        return $this->belongsTo(User::class, 'customer_id');
    }

    public function workspace()
    {
        return $this->belongsTo(\App\Models\Workspace::class, 'workspace_id');
    }

    // Helper methods for frontend
    public function getSenderName()
    {
        if ($this->sender_type === 'emp' && $this->senderEmployee) {
            $emp = $this->senderEmployee;
            if ($emp->relationLoaded('empPersonalDetails') && $emp->empPersonalDetails) {
                $details = $emp->empPersonalDetails;
                return trim(($details->first_name ?? '') . ' ' . ($details->middle_name ?? '') . ' ' . ($details->last_name ?? ''));
            }
            return $emp->employee_email ?? 'Unknown User';
        } elseif ($this->sender_type === 'user' && $this->senderUser) {
            return $this->senderUser->name ?? $this->senderUser->email ?? 'Unknown User';
        }
        return 'Unknown User';
    }

    public function getSenderImage()
    {
        if ($this->sender_type === 'emp' && $this->senderEmployee) {
            if ($this->senderEmployee->relationLoaded('tierEmpPersonalDetail') && $this->senderEmployee->tierEmpPersonalDetail) {
                return $this->senderEmployee->tierEmpPersonalDetail->image ?? '/img/profile.jpeg';
            }
            return '/img/profile.jpeg';
        } elseif ($this->sender_type === 'user' && $this->senderUser) {
            return $this->senderUser->image ?? '/img/profile.jpeg';
        }
        return '/img/profile.jpeg';
    }
}
