<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Notification extends BaseModel
{
    use HasFactory;
    protected $table = "all_notifications";

    protected $fillable = [
        'title',
        'short_description',
        'sender_id',
        'sender_type',
        'receiver_id',
        'receiver_type',
        'type',
        'read',
        'customer_id',
        'workspace_id'
    ];

    protected $hidden = [
        'customer_id',
        'workspace_id',
    ];

    protected $appends = [
        'sender_details',
        'receiver_details',
    ];

    /**
     * Get sender details based on sender_type
     * Returns data from EmpPersonalDetails if type is 'emp', or User if type is 'customer'
     */
    public function getSenderDetailsAttribute()
    {
        if (!$this->sender_id || !$this->sender_type) {
            return null;
        }

        if ($this->sender_type === 'emp') {
            $details = EmpPersonalDetails::where('emp_id', $this->sender_id)
                ->select('emp_id', 'first_name', 'middle_name', 'last_name', 'image')
                ->first();
            
            if ($details) {
                return [
                    'emp_id' => $details->emp_id,
                    'first_name' => $details->first_name,
                    'middle_name' => $details->middle_name,
                    'last_name' => $details->last_name,
                    'image' => $details->image,
                ];
            }
        } elseif ($this->sender_type === 'customer' || $this->sender_type === 'admin') {
            $user = User::where('id', $this->sender_id)
                ->select('id', 'name', 'avatar')
                ->first();
            
            if ($user) {
                return [
                    'emp_id' => $user->id,
                    'first_name' => $user->name,
                    'middle_name' => null,
                    'last_name' => null,
                    'image' => $user->avatar ?? 'upload/images/default.png',
                ];
            }
        }

        return null;
    }

    /**
     * Get receiver details based on receiver_type
     * Returns data from EmpPersonalDetails if type is 'emp', or User if type is 'customer'
     */
    public function getReceiverDetailsAttribute()
    {
        if (!$this->receiver_id || !$this->receiver_type) {
            return null;
        }

        if ($this->receiver_type === 'emp') {
            $details = EmpPersonalDetails::where('emp_id', $this->receiver_id)
                ->select('emp_id', 'first_name', 'middle_name', 'last_name', 'image')
                ->first();
            
            if ($details) {
                return [
                    'emp_id' => $details->emp_id,
                    'first_name' => $details->first_name,
                    'middle_name' => $details->middle_name,
                    'last_name' => $details->last_name,
                    'image' => $details->image,
                ];
            }
        } elseif ($this->receiver_type === 'customer' || $this->receiver_type === 'admin') {
            $user = User::where('id', $this->receiver_id)
                ->select('id', 'name', 'avatar')
                ->first();
            
            if ($user) {
                return [
                    'emp_id' => $user->id,
                    'first_name' => $user->name,
                    'middle_name' => null,
                    'last_name' => null,
                    'image' => $user->avatar ?? 'upload/images/default.png',
                ];
            }
        }

        return null;
    }

    /**
     * Override toArray to ensure camelCase naming for senderDetails and receiverDetails
     * This maintains backward compatibility with the old relationship names
     */
    public function toArray()
    {
        $array = parent::toArray();
        
        // Map snake_case accessor names to camelCase for backward compatibility
        if (isset($array['sender_details'])) {
            $array['senderDetails'] = $array['sender_details'];
            unset($array['sender_details']);
        }
        
        if (isset($array['receiver_details'])) {
            $array['receiverDetails'] = $array['receiver_details'];
            unset($array['receiver_details']);
        }
        
        return $array;
    }


    // abc
    public function sites()
    {
        return $this->belongsTo(Sites::class, 'site_id')->select('id', 'title');
    }

    public function teams()
    {
        return $this->belongsTo(EmpTeam::class, 'team_id')->select('id', 'title');
    }
}
